<?php
defined( 'ABSPATH' ) || die();

require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_Helper.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Exam.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Student.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Record.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/vendor/autoload.php';

use Razorpay\Api\Api as RazorpayApi;

class OEM_P_Payment {

	public static function payment() {
		OEM_Helper::check_nonce();

		try {
			ob_start();
			global $wpdb;

			$errors = array();

			$record_id = get_current_user_id();

			$record = OEM_M_Record::find( $record_id );

			if ( ! $record ) {
				throw new Exception( esc_html__( 'Registration not found.', 'online-exam-management' ) );
			}

			$exams            = ( isset( $_POST['exams'] ) && is_array( $_POST['exams'] ) ) ? array_map( 'absint', $_POST['exams'] ) : array();
			$payment_method   = isset( $_POST['payment_method'] ) ? sanitize_text_field( $_POST['payment_method'] ) : '';
			$current_page_url = isset( $_POST['current_page_url'] ) ? esc_url( $_POST['current_page_url'] ) : '';

			if ( empty( $exams ) ) {
				$errors['exams[]'] = esc_html__( 'Please select at least one exam', 'online-exam-management' );
				wp_send_json_error( $errors );
			}

			$exams_for_payment = OEM_M_Record::get_upcoming_exams_for_payment();

			$exams_id_for_payment = array();
			$total_amount         = 0;
			foreach ( $exams_for_payment as $exam ) {
				array_push( $exams_id_for_payment, $exam->ID );

				if ( in_array( $exam->ID, $exams ) ) {
					$total_amount += ( $exam->exam_fee );
				}
			}

			if ( $total_amount > 0 ) {
				if ( ! $payment_method ) {
					throw new Exception( esc_html__( 'Please select a payment method.', 'online-exam-management' ) );
				} elseif ( ! in_array( $payment_method, array_keys( OEM_Helper::payment_methods() ) ) ) {
					throw new Exception( esc_html__( 'Please select a valid payment method.', 'online-exam-management' ) );
				}
			}

			if ( array_diff( $exams, $exams_id_for_payment ) ) {
				$errors['exams[]'] = esc_html__( 'Please select valid exam(s).', 'online-exam-management' );
				wp_send_json_error( $errors );
			}

			OEM_Helper::check_buffer();

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}

		if ( count( $errors ) > 0 ) {
			wp_send_json_error( $errors );
		}

		try {
			ob_start();

			$wpdb->query( 'BEGIN;' );

			$now = OEM_Helper::now();

			$reg_number   = $record->oem_reg_number;
			$name         = $record->oem_name;
			$phone        = $record->oem_phone;
			$address      = $record->oem_address;
			$father_name  = $record->oem_father_name;
			$father_phone = $record->oem_father_phone;
			$photo_id     = $record->oem_photo_id;
			$login_email  = $record->user_email;
			$username     = $record->user_login;

			$gender    = $record->oem_gender;
			$dob       = $record->oem_dob;
			$id_number = $record->oem_id_number;

			$json = json_encode( array() );

			if ( $total_amount > 0 ) {

				$currency = OEM_Helper::currency();

				$receipt = OEM_Helper::generate_receipt_number();

				// Create payment receipt.
				$payment_data = array(
					'amount'         => $total_amount,
					'payment_method' => $payment_method,
					'currency'       => $currency,
					'receipt'        => $receipt,
					'record_user_id' => $record_id,
					'access_type'    => 'specified',
					'created_at'     => $now,
				);

				$success = $wpdb->insert( OEM_PAYMENTS, $payment_data );

				$payment_id = $wpdb->insert_id;

				OEM_Helper::check_buffer();

				if ( false === $success ) {
					throw new Exception( $wpdb->last_error );
				}

				// Insert exam payment.
				foreach ( $exams as $exam_id ) {
					$exam_payment_data = array(
						'exam_id'    => $exam_id,
						'payment_id' => $payment_id,
						'created_at' => $now,
					);

					$success = $wpdb->insert( OEM_EXAM_PAYMENT, $exam_payment_data );

					OEM_Helper::check_buffer();

					if ( false === $success ) {
						throw new Exception( $wpdb->last_error );
					}
				}

				if ( 'razorpay' === $payment_method ) {
					// Razorpay settings.
					$razorpay        = OEM_Helper::razorpay();
					$razorpay_enable = $razorpay['enable'];
					$razorpay_key    = $razorpay['key'];
					$razorpay_secret = $razorpay['secret'];

					// Create order.
					$api = new RazorpayApi( $razorpay_key, $razorpay_secret );

					$order = $api->order->create(
						array(
							'receipt'         => $receipt,
							'amount'          => $total_amount * 100,
							'payment_capture' => 1,
							'currency'        => $currency,
							'notes'           => array(
								'registration_number' => $reg_number,
								'name'                => $name,
								'phone'               => $phone,
								'address'             => $address,
								'username'            => $username,
							),
						)
					);

					// Store order_id.
					$success = $wpdb->update( OEM_PAYMENTS, array( 'order_id' => $order->id ), array( 'ID' => $payment_id ) );

					OEM_Helper::check_buffer();

					if ( false === $success ) {
						throw new Exception( $wpdb->last_error );
					}

					$json = json_encode(
						array(
							'amount'         => $total_amount,
							'payment_method' => $payment_method,
							'currency'       => $currency,
							'razorpay_key'   => $razorpay_key,
							'amount_x_100'   => $total_amount,
							'title'          => esc_attr__( 'Exam Fee Payment', 'online-exam-management' ),
							'description'    => esc_attr__( 'Exam fee payment.', 'online-exam-management' ),
							'image'          => esc_url( get_custom_logo() ),
							'order_id'       => esc_attr( $order->id ),
							'name'           => esc_attr( $name ),
							'login_email'    => esc_attr( $login_email ),
							'phone'          => esc_attr( $phone ),
							'address'        => esc_attr( $address ),
							'gender'         => esc_attr( $gender ),
							'dob'            => esc_attr( $dob ),
						)
					);
				}
				if ( 'stripe' === $payment_method ) {
					// stripe settings.
					$stripe        = OEM_Helper::stripe();
					$stripe_enable = $stripe['enable'];
					$stripe_key    = $stripe['key'];
					$stripe_secret = $stripe['secret'];

					// Store order_id.
					$success = $wpdb->update( OEM_PAYMENTS, array( 'order_id' => $payment_id ), array( 'ID' => $payment_id ) );

					OEM_Helper::check_buffer();

					if ( ! $stripe_enable ) {
						wp_send_json_error( esc_html__( 'Stripe payment method is currently unavailable.', 'online-exam-management' ) );
					}

					$pay_with_stripe_text = sprintf(
						/* translators: %s: amount to pay */
						__( 'Pay Amount %s using Stripe', 'online-exam-management' ),
						esc_html( OEM_Helper::get_money_text( $total_amount ) )
					);

					$html           = "<button class='oem-mt-2 float-md-right button btn btn-success' id='oem-stripe-btn'>$pay_with_stripe_text</button>";
					$amount_in_cent = $total_amount * 100;
					$json           = json_encode(
						array(
							'amount'         => $total_amount,
							'payment_method' => $payment_method,
							'currency'       => $currency,
							'stripe_key'     => $stripe_key,
							'amount_x_100'   => $amount_in_cent,
							'title'          => esc_attr__( 'Exam Fee Payment', 'online-exam-management' ),
							'description'    => esc_attr__( 'Exam fee payment.', 'online-exam-management' ),
							'image'          => esc_url( get_custom_logo() ),
							'order_id'       => esc_attr( $payment_id ),
							'name'           => esc_attr( $name ),
							'login_email'    => esc_attr( $login_email ),
							'phone'          => esc_attr( $phone ),
							'address'        => esc_attr( $address ),
						)
					);
				}
			} else {
				// Create student record for exam.

				foreach ( $exams as $exam_id ) {
					$record_exams_id = OEM_M_Record::get_record_students_exams_id( $record_id );
					// echo '<pre>' . var_dump($record_exams_id) . '</pre>';
					// Insert only if there is no record in this exam.
					if ( ! in_array( $exam_id, [$record_exams_id] ) ) {
						$roll_number = OEM_Helper::generate_roll_number( $exam_id );

						$student_data = array(
							'name'           => $name,
							'roll_number'    => $roll_number,
							'phone'          => $phone,
							'address'        => $address,
							'father_name'    => $father_name,
							'father_phone'   => $father_phone,
							'exam_id'        => $exam_id,
							'record_user_id' => $record_id,
							'user_id'        => $record_id,
							'gender'         => $gender,
							'dob'            => $dob,
							'id_number'      => $id_number,
						);

						$student_data['created_at'] = $now;

						$success = $wpdb->insert( OEM_STUDENTS, $student_data );

						if ( $success ) {

							$data = array(
								'exam_id' => $exam_id,
								'user_id' => $record_id,
							);
							wp_schedule_single_event( time() + 30, 'oem_notify_for_student_admission', $data );
						}

						// OEM_Helper::check_buffer();

						if ( false === $success ) {
							throw new Exception( $wpdb->last_error );
						}
					}
				}

				$json = json_encode(
					array(
						'free'    => true,
						'message' => esc_attr__( 'Registration for upcoming exam has been completed.', 'online-exam-management' ),
					)
				);
			}
			// $html = ob_get_clean();

			$wpdb->query( 'COMMIT;' );

			wp_send_json_success(
				array(
					'html' => $html,
					'json' => $json,
				)
			);
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}

	public static function razorpay_handler() {
		 // var_dump($_POST); die;

		OEM_Helper::check_nonce();

		try {
			global $wpdb;

			$wpdb->query( 'BEGIN;' );

			$now = OEM_Helper::now();

			$error_code = isset( $_POST['error_code'] ) ? sanitize_text_field( $_POST['error_code'] ) : '';

			if ( ! empty( $error_code ) ) {
				$error_description = isset( $_POST['error_description'] ) ? sanitize_text_field( $_POST['error_description'] ) : '';
				$error_field       = isset( $_POST['error_field'] ) ? sanitize_text_field( $_POST['error_field'] ) : '';
				throw new Exception( $error_description );
			}

			$razorpay_payment_id = isset( $_POST['razorpay_payment_id'] ) ? sanitize_text_field( $_POST['razorpay_payment_id'] ) : '';
			$razorpay_order_id   = isset( $_POST['razorpay_order_id'] ) ? sanitize_text_field( $_POST['razorpay_order_id'] ) : '';
			$razorpay_signature  = isset( $_POST['razorpay_signature'] ) ? sanitize_text_field( $_POST['razorpay_signature'] ) : '';

			if ( ! $razorpay_payment_id || ! $razorpay_order_id || ! $razorpay_signature ) {
				throw new Exception( esc_html__( 'Unable to verify the payment.', 'online-exam-management' ) );
			}

			// Razorpay settings.
			$razorpay        = OEM_Helper::razorpay();
			$razorpay_key    = $razorpay['key'];
			$razorpay_secret = $razorpay['secret'];

			$api        = new RazorpayApi( $razorpay_key, $razorpay_secret );
			$attributes = array(
				'razorpay_payment_id' => $razorpay_payment_id,
				'razorpay_order_id'   => $razorpay_order_id,
				'razorpay_signature'  => $razorpay_signature,
			);

			$order = $api->utility->verifyPaymentSignature( $attributes );

			$payment = OEM_M_Payment::fetch_payment_by_order_id( $razorpay_order_id, 'razorpay' );
			if ( ! $payment ) {
				throw new Exception( esc_html__( 'Payment not found.', 'online-exam-management' ) );
			}

			$payment_id = $payment->ID;

			$record = OEM_M_Record::find( $payment->record_user_id );
			if ( ! $record ) {
				throw new Exception( esc_html__( 'Registration not found.', 'online-exam-management' ) );
			}

			$record_id    = $record->ID;
			$reg_number   = $record->oem_reg_number;
			$name         = $record->oem_name;
			$phone        = $record->oem_phone;
			$address      = $record->oem_address;
			$father_name  = $record->oem_father_name;
			$father_phone = $record->oem_father_phone;
			$photo_id     = $record->oem_photo_id;
			$login_email  = $record->user_email;
			$username     = $record->user_login;

			if ( 'specified' === $payment->access_type ) {
				$exams = OEM_M_Payment::get_payment_exams( $payment_id );

				$record_exams_id = OEM_M_Record::get_record_students_exams_id( $record_id );

				foreach ( $exams as $exam ) {
					$exam_id = $exam->ID;

					// Insert only if there is no record in this exam.
					if ( ! in_array( $exam_id, $record_exams_id ) ) {
						$roll_number = OEM_Helper::generate_roll_number( $exam_id );

						$student_data = array(
							'name'           => $name,
							'roll_number'    => $roll_number,
							'phone'          => $phone,
							'address'        => $address,
							'father_name'    => $father_name,
							'father_phone'   => $father_phone,
							'exam_id'        => $exam_id,
							'record_user_id' => $record_id,
							'user_id'        => $record_id,
						);

						$student_data['created_at'] = $now;

						$success = $wpdb->insert( OEM_STUDENTS, $student_data );

						OEM_Helper::check_buffer();

						if ( false === $success ) {
							throw new Exception( $wpdb->last_error );
						}
					}
				}
			}

			$payment_data = array(
				'transaction_id' => $razorpay_payment_id,
				'status'         => 'completed',
				'updated_at'     => $now,
				'payment_date'   => $now,
			);

			$success = $wpdb->update( OEM_PAYMENTS, $payment_data, array( 'ID' => $payment_id ) );

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			$message = esc_html__( 'Payment made successfully.', 'online-exam-management' );

			wp_send_json_success( array( 'message' => $message ) );
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}
	public static function stripe_handler() {
		// var_dump($_POST); die;
		OEM_Helper::check_nonce();

			global $wpdb;

			$wpdb->query( 'BEGIN;' );

			$now = OEM_Helper::now();

			$error_code = isset( $_POST['error_code'] ) ? sanitize_text_field( $_POST['error_code'] ) : '';

		if ( ! empty( $error_code ) ) {
			$error_description = isset( $_POST['error_description'] ) ? sanitize_text_field( $_POST['error_description'] ) : '';
			$error_field       = isset( $_POST['error_field'] ) ? sanitize_text_field( $_POST['error_field'] ) : '';
			throw new Exception( $error_description );
		}

			// $stripe_payment_id = isset( $_POST['stripe_payment_id'] ) ? sanitize_text_field( $_POST['stripe_payment_id'] ) : '';
			$stripe_order_id = isset( $_POST['order_id'] ) ? sanitize_text_field( $_POST['order_id'] ) : '';
			$stripeToken     = isset( $_POST['stripeToken'] ) ? sanitize_text_field( $_POST['stripeToken'] ) : '';
			$amount          = isset( $_POST['amount'] ) ? sanitize_text_field( $_POST['amount'] ) : '';
			$currency        = get_option( 'oem_currency' );
		if ( ! $stripe_order_id || ! $stripeToken ) {
			throw new Exception( esc_html__( 'Unable to verify the payment.', 'online-exam-management' ) );
		}

			// stripe settings.
			$stripe        = OEM_Helper::stripe();
			$stripe_key    = $stripe['key'];
			$stripe_secret = $stripe['secret'];

			$payment = OEM_M_Payment::fetch_payment_by_order_id( $stripe_order_id, 'stripe' );
			$record  = OEM_M_Record::find( $payment->record_user_id );
		try {
			\Stripe\Stripe::setApiKey( $stripe_secret );
			$charge   = \Stripe\Charge::create(
				array(
					'amount'      => $amount,
					'currency'    => $currency,
					'description' => 'Exam',
					'source'      => $stripeToken,
				)
			);
			$customer = \Stripe\Customer::create(
				array(
					'name'    => '',
					'address' => array(
						'city'        => '',
						'country'     => '',
						'line1'       => '',
						'line2'       => '',
						'postal_code' => '',
						'state'       => '',
					),
				)
			);
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}

		if ( ! $payment ) {
			throw new Exception( esc_html__( 'Payment not found.', 'online-exam-management' ) );
		}

		if ( ! ( $charge && $charge->captured ) ) {
			wp_send_json_error( esc_html__( 'Unable to capture the payment.', 'school-management' ) );
		}

			$payment_id = $payment->ID;

		if ( ! $record ) {
			throw new Exception( esc_html__( 'Registration not found.', 'online-exam-management' ) );
		}

			$record_id    = $record->ID;
			$reg_number   = $record->oem_reg_number;
			$name         = $record->oem_name;
			$phone        = $record->oem_phone;
			$address      = $record->oem_address;
			$father_name  = $record->oem_father_name;
			$father_phone = $record->oem_father_phone;
			$photo_id     = $record->oem_photo_id;
			$login_email  = $record->user_email;
			$username     = $record->user_login;

		if ( 'specified' === $payment->access_type ) {
			$exams = OEM_M_Payment::get_payment_exams( $payment_id );

			$record_exams_id = OEM_M_Record::get_record_students_exams_id( $record_id );

			foreach ( $exams as $exam ) {
				$exam_id = $exam->ID;

				// Insert only if there is no record in this exam.
				if ( ! in_array( $exam_id, $record_exams_id ) ) {
					$roll_number = OEM_Helper::generate_roll_number( $exam_id );

					$student_data = array(
						'name'           => $name,
						'roll_number'    => $roll_number,
						'phone'          => $phone,
						'address'        => $address,
						'father_name'    => $father_name,
						'father_phone'   => $father_phone,
						'exam_id'        => $exam_id,
						'record_user_id' => $record_id,
						'user_id'        => $record_id,
					);

					$student_data['created_at'] = $now;

					$success = $wpdb->insert( OEM_STUDENTS, $student_data );

					OEM_Helper::check_buffer();

					if ( false === $success ) {
						throw new Exception( $wpdb->last_error );
					}
				}
			}
		}

			$payment_data = array(
				'transaction_id' => $stripe_order_id,
				'status'         => 'completed',
				'updated_at'     => $now,
				'payment_date'   => $now,
			);

			$success = $wpdb->update( OEM_PAYMENTS, $payment_data, array( 'ID' => $payment_id ) );

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			$message = esc_html__( 'Payment made successfully.', 'online-exam-management' );

			wp_send_json_success( array( 'message' => $message ) );
	}
}

