<?php
defined( 'ABSPATH' ) || die();

require_once OEM_PLUGIN_DIR_PATH . 'admin/inc/OEM_LM.php';

class OEM_Helper {

	public static function check_demo_mode() {
		if ( OEM_DEMO_MODE ) {
			wp_send_json_error( OEM_DEMO_MODE_MESSAGE );
		}
	}

	public static function get_default_date_format() {
		return 'd-m-Y';
	}

	public static function get_default_time_format() {
		return 'h:i a';
	}

	public static function get_default_at_format() {
		return 'd-m-Y h:i a';
	}

	public static function get_default_at_format_r() {
		return 'd-m-Y h:i a';
	}

	public static function date_format() {
		$date_format = get_option( 'oem_date_format' );
		if ( ! $date_format ) {
			$date_format = self::get_default_date_format();
		}

		return $date_format;
	}

	public static function time_format() {
		$time_format = get_option( 'oem_time_format' );
		if ( ! $time_format ) {
			$time_format = self::get_default_time_format();
		}

		return $time_format;
	}

	public static function at_format() {
		$at_format = get_option( 'oem_at_format' );
		if ( ! $at_format ) {
			$at_format = self::get_default_at_format();
		}

		return $at_format;
	}
	public static function at_format_r() {
		$at_format = get_option( 'oem_at_format' );
		if ( ! $at_format ) {
			$at_format = self::get_default_date_format();
		}

		return $at_format;
	}

	public static function get_date_text( $date, $empty = '' ) {
		if ( $date && $date !== '0000-00-00 00:00:00' ) {
			return date_format( date_create( $date ), self::date_format() );
		}
		return $empty;
	}

	public static function get_time_text( $time, $empty = '' ) {
		if ( $time ) {
			return date_format( date_create( $time ), self::get_default_time_format() );
		}
		return $empty;
	}

	public static function get_at_text( $at, $empty = '' ) {
		if ( $at ) {
			return date_format( date_create( $at ), self::at_format() );
		}
		return $empty;
	}

	public static function get_at_text_r( $at, $empty = '' ) {
		if ( $at ) {
			return date_format( date_create( $at ), self::at_format_r() );
		}
		return $empty;
	}

	public static function get_at_text_from_format( $at, $format, $empty = '' ) {
		if ( $at ) {
			return date_format( date_create( $at ), $format );
		}
		return $empty;
	}

	public static function at_formats() {
		return array(
			'd-m-Y h:i a',
			'd/m/Y h:i a',
			'Y-m-d h:i a',
			'Y/m/d h:i a',
			'm-d-Y h:i a',
			'm/d/Y h:i a',
			'd-m-Y h:i A',
			'd/m/Y h:i A',
			'Y-m-d h:i A',
			'Y/m/d h:i A',
			'm-d-Y h:i A',
			'm/d/Y h:i A',
			'd-m-Y H:i',
			'd/m/Y H:i',
			'Y-m-d H:i',
			'Y/m/d H:i',
			'm-d-Y H:i',
			'm/d/Y H:i',
		);
	}

	public static function stripslashes( $text, $empty = '' ) {
		if ( $text ) {
			return stripcslashes( $text );
		}
		return $empty;
	}

	public static function get_text( $text, $empty = '' ) {
		if ( $text ) {
			return $text;
		}
		return $empty;
	}

	public static function get_status_text( $is_active, $color = false ) {
		if ( $is_active ) {
			if ( $color ) {
				return '<span class="oem-font-bold oem-text-success">' . self::get_active_text() . '</span>';
			}
			return self::get_active_text();
		}

		if ( $color ) {
			return '<span class="oem-font-bold oem-text-danger">' . self::get_inactive_text() . '</span>';
		}
		return self::get_inactive_text();
	}

	public static function get_active_text() {
		return esc_html__( 'Active', 'online-exam-management' );
	}

	public static function get_inactive_text() {
		return esc_html__( 'Inactive', 'online-exam-management' );
	}

	public static function get_yes_no_text( $is_yes, $color = false ) {
		if ( $is_yes ) {
			if ( $color ) {
				return '<span class="oem-font-bold oem-text-success">' . self::get_yes_text() . '</span>';
			}
			return self::get_yes_text();
		}

		if ( $color ) {
			return '<span class="oem-font-bold oem-text-danger">' . self::get_no_text() . '</span>';
		}
		return self::get_no_text();
	}

	public static function get_yes_text() {
		 return esc_html__( 'Yes', 'online-exam-management' );
	}

	public static function get_no_text() {
		return esc_html__( 'No', 'online-exam-management' );
	}

	public static function get_exam_status_text( $exam_at, $duration, $color = false, $get_key = false ) {
		$now = new DateTime( self::now() );

		$duration = absint( $duration );

		$start_at = new DateTime( $exam_at );

		$end_at = new DateTime( $exam_at );
		$end_at = $end_at->modify( '+' . $duration . ' minutes' );

		if ( $now > $end_at ) {
			$key = 'completed';
		} elseif ( $now >= $start_at && $now <= $end_at ) {
			$key = 'active';
		} else {
			$key = 'pending';
		}

		$text = self::get_exam_status( $key );

		if ( $color ) {
			$text = self::get_exam_status_html( $key, $text );
		}

		if ( $get_key ) {
			return array(
				'key'  => $key,
				'text' => $text,
			);
		}

		return $text;
	}

	public static function get_exam_status_html( $key, $text ) {
		return '<span class="oem-font-bold oem-text-' . self::get_exam_status_color( $key ) . '">' . esc_html( $text ) . '</span>';
	}

	public static function get_exam_statuses() {
		return array(
			'pending'   => esc_html__( 'To be Started', 'online-exam-management' ),
			'active'    => esc_html__( 'On-going', 'online-exam-management' ),
			'completed' => esc_html__( 'Completed', 'online-exam-management' ),
		);
	}

	public static function get_exam_status( $key ) {
		return self::get_exam_statuses()[ $key ];
	}

	public static function get_exam_status_colors() {
		return array(
			'pending'   => 'danger',
			'active'    => 'primary',
			'completed' => 'success',
		);
	}

	public static function get_exam_status_color( $key ) {
		return self::get_exam_status_colors()[ $key ];
	}

	public static function sanitize_marks( $marks, $empty_if_zero = false, $empty = '' ) {
		if ( $empty_if_zero && ! $marks ) {
			return $empty;
		}
		return abs( (float) $marks );
	}

	public static function sanitize_percentage( $total, $obtained ) {
		if ( ! $total ) {
			return '';
		}
		return (float) number_format( ( $obtained * 100 ) / $total, 2, '.', '' );
	}

	public static function get_percentage_text( $total, $obtained ) {
		if ( ! $total ) {
			return '-';
		}
		return number_format( ( $obtained * 100 ) / $total, 2, '.', '' ) . ' %';
	}

	public static function limit_string( $content, $number_of_characters = 100 ) {
		if ( strlen( $content ) > $number_of_characters ) {
			$position = strpos( $content, ' ', $number_of_characters );
			$dots     = '...';
			return substr( $content, 0, $position ) . $dots;
		}

		return $content;
	}

	public static function check_buffer( $show_buffer_error = true ) {
		$buffer = ob_get_clean();
		if ( ! empty( $buffer ) ) {
			if ( $show_buffer_error ) {
				throw new Exception( $buffer );
			}

			throw new Exception( esc_html__( 'Unexpected error occurred!', 'online-exam-management' ) );
		}
	}

	public static function check_nonce( $action = 'oem_nonce' ) {
		if ( ! wp_verify_nonce( $_POST[ $action ], $action ) ) {
			die();
		}
	}

	public static function get_page_url( $page = '' ) {
		if ( defined( 'OEM_MENU_' . strtoupper( $page ) ) ) {
			return admin_url( 'admin.php?page=' . constant( 'OEM_MENU_' . strtoupper( $page ) ) );
		}
		return '';
	}

	public static function question_types() {
		return array(
			'obj_single' => esc_html__( 'Objective: Single-Choice', 'online-exam-management' ),
			'obj_multi'  => esc_html__( 'Objective: Multi-Choice', 'online-exam-management' ),
			'subjective' => esc_html__( 'Subjective', 'online-exam-management' ),
		);
	}

	public static function is_objective( $type ) {
		if ( in_array( $type, array( 'obj_single', 'obj_multi' ) ) ) {
			return true;
		}

		return false;
	}

	public static function get_subjective_marks_status( $key, $get_keys = false, $color = true ) {
		if ( $key === 'complete' ) {
			$text = esc_html__( 'No Action Required', 'online-exam-management' );
			if ( $color ) {
				$text = ( '<span class="oem-text-success oem-font-bold">' . $text . '</span>' );
			}
		} else {
			$key = 'incomplete';

			$text = esc_html__( 'Action Required', 'online-exam-management' );
			if ( $color ) {
				$text = ( '<span class="oem-text-danger oem-font-bold">' . $text . '</span>' );
			}
		}

		if ( $get_keys ) {
			return array(
				'key'  => $key,
				'text' => $text,
			);
		}

		return $text;
	}

	public static function question_type( $type ) {
		if ( isset( self::question_types()[ $type ] ) ) {
			return self::question_types()[ $type ];
		}
	}

	public static function get_option_number( $number ) {
		return self::get_numeric_option_number( $number );
	}

	public static function get_numeric_option_number( $number ) {
		return absint( $number );
	}

	public static function filter_by_date( $search_value ) {
		$date   = DateTime::createFromFormat( self::date_format(), $search_value );
		$format = '';

		if ( $date ) {
			$format = 'Y-m-d';
		} else {
			if ( 'd-m-Y' === self::date_format() ) {
				$date   = DateTime::createFromFormat( 'm-Y', $search_value );
				$format = 'Y-m';
			} elseif ( 'd/m/Y' === self::date_format() ) {
				$date   = DateTime::createFromFormat( 'm/Y', $search_value );
				$format = 'Y-m';
			} elseif ( 'Y-m-d' === self::date_format() ) {
				$date   = DateTime::createFromFormat( 'Y-m', $search_value );
				$format = 'Y-m';
			} elseif ( 'Y/m/d' === self::date_format() ) {
				$date   = DateTime::createFromFormat( 'Y/m', $search_value );
				$format = 'Y-m';
			}

			if ( ! $date ) {
				$date   = DateTime::createFromFormat( 'Y', $search_value );
				$format = 'Y';
			}
		}

		return array(
			'date'   => $date,
			'format' => $format,
		);
	}

	public static function now() {
		return current_time( 'Y-m-d H:i:s' );
	}

	public static function get_image_mime() {
		return array( 'image/jpg', 'image/jpeg', 'image/png' );
	}

	public static function get_csv_mime() {
		 return array( 'text/x-comma-separated-values', 'text/comma-separated-values', 'application/octet-stream', 'application/vnd.ms-excel', 'application/x-csv', 'text/x-csv', 'text/csv', 'application/csv', 'application/excel', 'application/vnd.msexcel', 'text/plain' );
	}

	public static function is_valid_file( $file, $type = 'image' ) {
		$get_mime = 'get_' . $type . '_mime';

		if ( extension_loaded( 'fileinfo' ) ) {
			$finfo = finfo_open( FILEINFO_MIME_TYPE );
			$mime  = finfo_file( $finfo, $file['tmp_name'] );
			finfo_close( $finfo );
		} else {
			$mime = $file['type'];
		}

		if ( ! in_array( $mime, self::$get_mime() ) ) {
			return false;
		}

		return true;
	}

	public static function get_default_subject() {
		return esc_html__( 'Miscellaneous', 'online-exam-management' );
	}

	public static function export_and_close_csv_file( $f, $filename ) {
		 fseek( $f, 0 );

		header( 'Content-Type: text/html' );
		header( 'Content-Disposition: attachment; filename="' . $filename . '";' );

		fpassthru( $f );

		fclose( $f );

		exit;
	}

	public static function get_csv_error_msg( $row, $column, $space = ' ' ) {
		return $space . sprintf(
			wp_kses(
				/* translators: 1: CSV row number, 2: CSV column number */
				__( 'Error in CSV: <span class="oem-font-bold">row</span>: %1$s, <span class="oem-font-bold">column</span>: %2$s', 'online-exam-management' ),
				array(
					'span' => array(
						'class' => array(),
					),
				)
			),
			esc_html( $row ),
			esc_html( $column )
		);
	}

	public static function array_has_duplicates( $array ) {
		 return count( $array ) !== count( array_unique( $array ) );
	}

	public static function online_exam_page_url() {
		 $exam_page_url = get_option( 'oem_exam_page_url' );
		if ( ! $exam_page_url ) {
			return '';
		}

		return esc_url( $exam_page_url );
	}

	public static function registration_page_url() {
		$registration_page_url = get_option( 'oem_registration_page_url' );
		if ( ! $registration_page_url ) {
			return '';
		}

		return esc_url( $registration_page_url );
	}

	public static function enable_photo() {
		 $enable_photo = get_option( 'enable_photo' );
		if ( ! $enable_photo ) {
			return '';
		}

		return $enable_photo;
	}

	public static function currency() {
		 $currency = get_option( 'oem_currency' );
		if ( isset( self::currency_symbols()[ $currency ] ) ) {
			return $currency;
		}

		return self::get_default_currency();
	}

	public static function currency_symbol() {
		return self::currency_symbols()[ self::currency() ];
	}

	public static function sanitize_money( $money ) {
		return (float) number_format( max( (float) $money, 0 ), 2, '.', '' );
	}

	public static function get_money_text( $money ) {
		$money = number_format( (float) $money, 2, '.', '' );
		return self::currency_symbol() . number_format( (float) $money, 2, '.', '' );
	}

	public static function get_default_currency() {
		 return 'USD';
	}

	public static function currency_symbols() {
		return array(
			'AED' => '&#1583;.&#1573;',
			'AFN' => '&#65;&#102;',
			'ALL' => '&#76;&#101;&#107;',
			'ANG' => '&#402;',
			'AOA' => '&#75;&#122;',
			'ARS' => '&#36;',
			'AUD' => '&#36;',
			'AWG' => '&#402;',
			'AZN' => '&#1084;&#1072;&#1085;',
			'BAM' => '&#75;&#77;',
			'BBD' => '&#36;',
			'BDT' => '&#2547;',
			'BGN' => '&#1083;&#1074;',
			'BHD' => '.&#1583;.&#1576;',
			'BIF' => '&#70;&#66;&#117;',
			'BMD' => '&#36;',
			'BND' => '&#36;',
			'BOB' => '&#36;&#98;',
			'BRL' => '&#82;&#36;',
			'BSD' => '&#36;',
			'BTN' => '&#78;&#117;&#46;',
			'BWP' => '&#80;',
			'BYR' => '&#112;&#46;',
			'BZD' => '&#66;&#90;&#36;',
			'CAD' => '&#36;',
			'CDF' => '&#70;&#67;',
			'CHF' => '&#67;&#72;&#70;',
			'CLP' => '&#36;',
			'CNY' => '&#165;',
			'COP' => '&#36;',
			'CRC' => '&#8353;',
			'CUP' => '&#8396;',
			'CVE' => '&#36;',
			'CZK' => '&#75;&#269;',
			'DJF' => '&#70;&#100;&#106;',
			'DKK' => '&#107;&#114;',
			'DOP' => '&#82;&#68;&#36;',
			'DZD' => '&#1583;&#1580;',
			'EGP' => '&#163;',
			'ETB' => '&#66;&#114;',
			'EUR' => '&#8364;',
			'FJD' => '&#36;',
			'FKP' => '&#163;',
			'GBP' => '&#163;',
			'GEL' => '&#4314;',
			'GHS' => '&#162;',
			'GIP' => '&#163;',
			'GMD' => '&#68;',
			'GNF' => '&#70;&#71;',
			'GTQ' => '&#81;',
			'GYD' => '&#36;',
			'HKD' => '&#36;',
			'HNL' => '&#76;',
			'HRK' => '&#107;&#110;',
			'HTG' => '&#71;',
			'HUF' => '&#70;&#116;',
			'IDR' => '&#82;&#112;',
			'ILS' => '&#8362;',
			'INR' => '&#8377;',
			'IQD' => '&#1593;.&#1583;',
			'IRR' => '&#65020;',
			'ISK' => '&#107;&#114;',
			'JEP' => '&#163;',
			'JMD' => '&#74;&#36;',
			'JOD' => '&#74;&#68;',
			'JPY' => '&#165;',
			'KES' => '&#75;&#83;&#104;',
			'KGS' => '&#1083;&#1074;',
			'KHR' => '&#6107;',
			'KMF' => '&#67;&#70;',
			'KPW' => '&#8361;',
			'KRW' => '&#8361;',
			'KWD' => '&#1583;.&#1603;',
			'KYD' => '&#36;',
			'KZT' => '&#1083;&#1074;',
			'LAK' => '&#8365;',
			'LBP' => '&#163;',
			'LKR' => '&#8360;',
			'LRD' => '&#36;',
			'LSL' => '&#76;',
			'LTL' => '&#76;&#116;',
			'LVL' => '&#76;&#115;',
			'LYD' => '&#1604;.&#1583;',
			'MAD' => '&#1583;.&#1605;.',
			'MDL' => '&#76;',
			'MGA' => '&#65;&#114;',
			'MKD' => '&#1076;&#1077;&#1085;',
			'MMK' => '&#75;',
			'MNT' => '&#8366;',
			'MOP' => '&#77;&#79;&#80;&#36;',
			'MRO' => '&#85;&#77;',
			'MUR' => '&#8360;',
			'MVR' => '.&#1923;',
			'MWK' => '&#77;&#75;',
			'MXN' => '&#36;',
			'MYR' => '&#82;&#77;',
			'MZN' => '&#77;&#84;',
			'NAD' => '&#36;',
			'NGN' => '&#8358;',
			'NIO' => '&#67;&#36;',
			'NOK' => '&#107;&#114;',
			'NPR' => '&#8360;',
			'NZD' => '&#36;',
			'OMR' => '&#65020;',
			'PAB' => '&#66;&#47;&#46;',
			'PEN' => '&#83;&#47;&#46;',
			'PGK' => '&#75;',
			'PHP' => '&#8369;',
			'PKR' => '&#8360;',
			'PLN' => '&#122;&#322;',
			'PYG' => '&#71;&#115;',
			'QAR' => '&#65020;',
			'RON' => '&#108;&#101;&#105;',
			'RSD' => '&#1044;&#1080;&#1085;&#46;',
			'RUB' => '&#1088;&#1091;&#1073;',
			'RWF' => '&#1585;.&#1587;',
			'SAR' => '&#65020;',
			'SBD' => '&#36;',
			'SCR' => '&#8360;',
			'SDG' => '&#163;',
			'SEK' => '&#107;&#114;',
			'SGD' => '&#36;',
			'SHP' => '&#163;',
			'SLL' => '&#76;&#101;',
			'SOS' => '&#83;',
			'SRD' => '&#36;',
			'STD' => '&#68;&#98;',
			'SVC' => '&#36;',
			'SYP' => '&#163;',
			'SZL' => '&#76;',
			'THB' => '&#3647;',
			'TJS' => '&#84;&#74;&#83;',
			'TMT' => '&#109;',
			'TND' => '&#1583;.&#1578;',
			'TOP' => '&#84;&#36;',
			'TRY' => '&#8356;',
			'TTD' => '&#36;',
			'TWD' => '&#78;&#84;&#36;',
			'TZS' => '',
			'UAH' => '&#8372;',
			'UGX' => '&#85;&#83;&#104;',
			'USD' => '&#36;',
			'UYU' => '&#36;&#85;',
			'UZS' => '&#1083;&#1074;',
			'VEF' => '&#66;&#115;',
			'VND' => '&#8363;',
			'VUV' => '&#86;&#84;',
			'WST' => '&#87;&#83;&#36;',
			'XAF' => '&#70;&#67;&#70;&#65;',
			'XCD' => '&#36;',
			'XDR' => '',
			'XOF' => '',
			'XPF' => '&#70;',
			'YER' => '&#65020;',
			'ZAR' => '&#82;',
			'ZMK' => '&#90;&#75;',
			'ZWL' => '&#90;&#36;',
		);
	}

	public static function email_type() {
		return array(
			'wpmail' => esc_html__( 'WP Mail', 'online-exam-management' ),
			'smtp'   => esc_html__( 'SMTP', 'online-exam-management' ),
		);
	}

	public static function payment_status() {
		return array(
			'pending'   => esc_html__( 'Pending', 'online-exam-management' ),
			'completed' => esc_html__( 'Completed', 'online-exam-management' ),
		);
	}

	public static function get_payment_status_text( $key, $empty = '' ) {
		$all_payment_status = self::payment_status();

		if ( array_key_exists( $key, $all_payment_status ) ) {
			return $all_payment_status[ $key ];
		}

		return $empty;
	}

	public static function payment_methods() {
		return array(
			'razorpay' => esc_html__( 'Razorpay', 'online-exam-management' ),
			'stripe'   => esc_html__( 'Stripe', 'online-exam-management' ),
		);
	}

	public static function get_payment_method_text( $key, $empty = '' ) {
		$all_payment_methods = self::payment_methods();

		if ( array_key_exists( $key, $all_payment_methods ) ) {
			return $all_payment_methods[ $key ];
		}

		return $empty;
	}

	public static function razorpay() {
		 $razorpay = get_option( 'oem_razorpay' );
		if ( ! is_array( $razorpay ) ) {
			$razorpay = array();
		}

		$razorpay['enable'] = isset( $razorpay['enable'] ) ? (bool) $razorpay['enable'] : 0;
		$razorpay['key']    = isset( $razorpay['key'] ) ? $razorpay['key'] : '';
		$razorpay['secret'] = isset( $razorpay['secret'] ) ? $razorpay['secret'] : '';

		return $razorpay;
	}

	public static function stripe() {
		$stripe = get_option( 'oem_stripe' );
		if ( ! is_array( $stripe ) ) {
			$stripe = array();
		}

		$stripe['enable'] = isset( $stripe['enable'] ) ? (bool) $stripe['enable'] : 0;
		$stripe['key']    = isset( $stripe['key'] ) ? $stripe['key'] : '';
		$stripe['secret'] = isset( $stripe['secret'] ) ? $stripe['secret'] : '';

		return $stripe;
	}

	public static function get_email_settings() {
		$email = get_option( 'oem_email_settings_data' );
		if ( ! is_array( $email ) ) {
			$email = array();
		}

		$email['email_type']            = isset( $email['email_type'] ) ? $email['email_type'] : '';
		$email['email_from_name']       = isset( $email['email_from_name'] ) ? $email['email_from_name'] : '';
		$email['email_smtp_host']       = isset( $email['email_smtp_host'] ) ? $email['email_smtp_host'] : '';
		$email['email_smtp_username']   = isset( $email['email_smtp_username'] ) ? $email['email_smtp_username'] : '';
		$email['email_smtp_password']   = isset( $email['email_smtp_password'] ) ? $email['email_smtp_password'] : '';
		$email['email_smtp_encryption'] = isset( $email['email_smtp_encryption'] ) ? $email['email_smtp_encryption'] : '';
		$email['email_smtp_port']       = isset( $email['email_smtp_port'] ) ? $email['email_smtp_port'] : '';

		return $email;
	}

	public static function get_email_template() {
		$template = get_option( 'oem_email_template_data' );
		if ( ! is_array( $template ) ) {
			$template = array();
		}

		$template['registration_template']   = isset( $template['registration_template'] ) ? $template['registration_template'] : '';
		$template['registration_subject']    = isset( $template['registration_subject'] ) ? $template['registration_subject'] : '';
		$template['email_registration_body'] = isset( $template['email_registration_body'] ) ? $template['email_registration_body'] : '';
		$template['result_template']         = isset( $template['result_template'] ) ? $template['result_template'] : '';
		$template['result_subject']          = isset( $template['result_subject'] ) ? $template['result_subject'] : '';
		$template['email_result_body']       = isset( $template['email_result_body'] ) ? $template['email_result_body'] : '';

		return $template;
	}

	public static function get_user_data() {
		$template = get_option( 'oem_email_send' );
		if ( ! is_array( $template ) ) {
			$template = array();
		}
		$template['user_id'] = isset( $template['user_id'] ) ? $template['user_id'] : '';
		$template['exam_id'] = isset( $template['exam_id'] ) ? $template['exam_id'] : '';

		return $template;
	}

	public static function send_email( $to, $subject, $body, $name = '', $email_for = '', $placeholders = array() ) {

				$available_placeholders = array_keys( self::email_placeholder() );

				if ( isset( $available_placeholders ) ) {
					foreach ( $placeholders as $key => $value ) {
						if ( in_array( $key, $available_placeholders ) ) {
							$subject = str_replace( $key, $value, $subject );
							$body    = str_replace( $key, $value, $body );
						}
					}
				}

		$settings   = self::get_email_settings();
		$email_type = $settings['email_type'];

		if ( 'wp_mail' === $email_type ) {

			$from_name  = $settings['email_from_name'];
			$from_email = $settings['from_email'];

			if ( is_array( $to ) ) {
				foreach ( $to as $key => $value ) {
					$to[ $key ] = $name[ $key ] . ' <' . $value . '>';
				}
			} else {
				if ( ! empty( $name ) ) {
					$to = "$name <$to>";
				}
			}

			$headers = array();
			array_push( $headers, 'Content-Type: text/html; charset=UTF-8' );
			if ( ! empty( $from_name ) ) {
				array_push( $headers, "From: $from_name <$from_email>" );
			}

			$status = wp_mail( $to, html_entity_decode( $subject ), $body, $headers, array() );
			return $status;

		} elseif ( 'smtp' === $email_type ) {

			$smtp       = self::get_email_settings();
			$from_name  = $smtp['email_from_name'];
			$host       = $smtp['email_smtp_host'];
			$username   = $smtp['email_smtp_username'];
			$password   = $smtp['email_smtp_password'];
			$encryption = $smtp['email_smtp_encryption'];
			$port       = $smtp['email_smtp_port'];

			global $wp_version;

			require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
			require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
			require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';
			$mail = new PHPMailer\PHPMailer\PHPMailer( true );

			try {
				$mail->CharSet  = 'UTF-8';
				$mail->Encoding = 'base64';

				if ( $host && $port ) {
					$mail->IsSMTP();
					$mail->Host = $host;
					if ( ! empty( $username ) && ! empty( $password ) ) {
						$mail->SMTPAuth = true;
						$mail->Password = $password;
					} else {
						$mail->SMTPAuth = false;
					}
					if ( ! empty( $encryption ) ) {
						$mail->SMTPSecure = $encryption;
					} else {
						$mail->SMTPSecure = null;
					}
					$mail->Port = $port;
				}

				$mail->Username = $username;

				$mail->setFrom( $mail->Username, $from_name );

				$mail->Subject = html_entity_decode( $subject );
				$mail->Body    = $body;

				$mail->IsHTML( true );

				if ( is_array( $to ) ) {
					foreach ( $to as $key => $value ) {
						$mail->AddAddress( $value, $name[ $key ] );
					}
				} else {
					$mail->AddAddress( $to, $name );
				}

				$status = $mail->Send();
				return $status;

			} catch ( Exception $e ) {
			}

			return false;
		}
	}

	public static function email_placeholder() {
		return array(
			'[STUDENT_NAME]'   => esc_html__( 'Student Name', 'online-exam-management' ),
			'[ROLL_NUMBER]'    => esc_html__( 'Roll Number', 'online-exam-management' ),
			'[LOGIN_USERNAME]' => esc_html__( 'Login Username', 'online-exam-management' ),
			'[LOGIN_EMAIL]'    => esc_html__( 'Login Email Number', 'online-exam-management' ),
		);
	}


	public static function currency_supports_razorpay( $currency ) {
		if ( in_array( $currency, self::razorpay_supported_currencies() ) ) {
			return true;
		}
		return false;
	}

	public static function registration_settings() {
		$registration = get_option( 'oem_registration' );
		if ( ! is_array( $registration ) ) {
			$registration = array();
		}

		$registration['reg_no_prefix']     = isset( $registration['reg_no_prefix'] ) ? $registration['reg_no_prefix'] : '';
		$registration['reg_no_base']       = isset( $registration['reg_no_base'] ) ? absint( $registration['reg_no_base'] ) : 0;
		$registration['last_reg_no_index'] = isset( $registration['last_reg_no_index'] ) ? absint( $registration['last_reg_no_index'] ) : 0;

		return $registration;
	}

	public static function currency_supports_stripe( $currency ) {
		if ( in_array( $currency, self::stripe_supported_currencies() ) ) {
			return true;
		}
		return false;
	}

	public static function razorpay_supported_currencies() {
		return array( 'INR' );
	}
	public static function stripe_supported_currencies() {
		return array( 'AFN', 'ALL', 'DZD', 'AOA', 'ARS', 'AMD', 'AWG', 'AUD', 'AZN', 'BSD', 'BDT', 'BBD', 'BZD', 'BMD', 'BOB', 'BAM', 'BWP', 'BRL', 'GBP', 'BND', 'BGN', 'BIF', 'KHR', 'CAD', 'CVE', 'KYD', 'XAF', 'XPF', 'CLP', 'CNY', 'COP', 'KMF', 'CDF', 'CRC', 'HRK', 'CZK', 'DKK', 'DJF', 'DOP', 'XCD', 'EGP', 'ETB', 'EUR', 'FKP', 'FJD', 'GMD', 'GEL', 'GIP', 'GTQ', 'GNF', 'GYD', 'HTG', 'HNL', 'HKD', 'HUF', 'ISK', 'INR', 'IDR', 'ILS', 'JMD', 'JPY', 'KZT', 'KES', 'KGS', 'LAK', 'LBP', 'LSL', 'LRD', 'MOP', 'MKD', 'MGA', 'MWK', 'MYR', 'MVR', 'MRO', 'MUR', 'MXN', 'MDL', 'MNT', 'MAD', 'MZN', 'MMK', 'NAD', 'NPR', 'ANG', 'TWD', 'NZD', 'NIO', 'NGN', 'NOK', 'PKR', 'PAB', 'PGK', 'PYG', 'PEN', 'PHP', 'PLN', 'QAR', 'RON', 'RUB', 'RWF', 'STD', 'SHP', 'SVC', 'WST', 'SAR', 'RSD', 'SCR', 'SLL', 'SGD', 'SBD', 'SOS', 'ZAR', 'KRW', 'LKR', 'SRD', 'SZL', 'SEK', 'CHF', 'TJS', 'TZS', 'THB', 'TOP', 'TTD', 'TRY', 'UGX', 'UAH', 'AED', 'USD', 'UYU', 'UZS', 'VUV', 'VND', 'XOF', 'YER', 'ZMW' );
	}

	public static function payment_receipt_settings() {
		 $payment_receipt = get_option( 'oem_payment_receipt' );
		if ( ! is_array( $payment_receipt ) ) {
			$payment_receipt = array();
		}

		$payment_receipt['receipt_no_prefix']     = isset( $payment_receipt['receipt_no_prefix'] ) ? $payment_receipt['receipt_no_prefix'] : '';
		$payment_receipt['receipt_no_base']       = isset( $payment_receipt['receipt_no_base'] ) ? absint( $payment_receipt['receipt_no_base'] ) : 0;
		$payment_receipt['last_receipt_no_index'] = isset( $payment_receipt['last_receipt_no_index'] ) ? absint( $payment_receipt['last_receipt_no_index'] ) : 0;

		return $payment_receipt;
	}

	public static function get_result_status_text( $minimum_percentage, $obtained_percentage ) {
		if ( $obtained_percentage >= $minimum_percentage ) {
			return '<span class="oem-text-success">' . esc_html__( 'Passed', 'online-exam-management' ) . '</span>';
		}

		return '<span class="oem-text-danger">' . esc_html__( 'Failed', 'online-exam-management' ) . '</span>';
	}

	public static function generate_roll_number( $exam_id ) {
		global $wpdb;

		$exam = OEM_M_Exam::get_exam_roll_number_settings( $exam_id );

		$prefix     = $exam->roll_no_prefix;
		$base       = $exam->roll_no_base;
		$last_index = $exam->last_roll_no_index;

		$new_index_count = absint( $last_index ) + 1;

		// Update last index.
		$data = array(
			'last_roll_no_index' => $new_index_count,
		);

		// Roll number.
		$roll_number = ( $prefix . ( $new_index_count + $base ) );

		$success = $wpdb->update( OEM_EXAMS, $data, array( 'ID' => $exam_id ) );

		self::check_buffer();

		if ( false === $success ) {
			throw new Exception( $wpdb->last_error );
		}

		// Check if student already exists with this roll number in exam.
		$student_exists = OEM_M_Student::get_student_by_roll_number( $roll_number, $exam_id );
		if ( $student_exists ) {
			// Regenerate roll number.
			return self::generate_roll_number( $exam_id );
		} else {
			return $roll_number;
		}
	}

	public static function generate_reg_number() {
		$registration = self::registration_settings();
		$prefix       = $registration['reg_no_prefix'];
		$base         = $registration['reg_no_base'];
		$last_index   = $registration['last_reg_no_index'];

		$new_index_count = absint( $last_index ) + 1;

		// Update last index.
		$registration['last_reg_no_index'] = $new_index_count;
		update_option( 'oem_registration', $registration );

		// Registration number.
		$reg_number = ( $prefix . ( $new_index_count + $base ) );

		// Check if record already exists with this registration number.
		$record_exists = OEM_M_Record::get_record_by_reg_number( $reg_number );
		if ( $record_exists ) {
			// Regenerate registration number.
			return self::generate_reg_number();
		} else {
			return $reg_number;
		}
	}

	public static function generate_receipt_number() {
		$payment_receipt = self::payment_receipt_settings();
		$prefix          = $payment_receipt['receipt_no_prefix'];
		$base            = $payment_receipt['receipt_no_base'];
		$last_index      = $payment_receipt['last_receipt_no_index'];

		$new_index_count = absint( $last_index ) + 1;

		// Update last index.
		$payment_receipt['last_receipt_no_index'] = $new_index_count;
		update_option( 'oem_payment_receipt', $payment_receipt );

		// Payment Receipt number.
		$receipt_number = ( $prefix . ( $new_index_count + $base ) );

		// Check if payment already exists with this receipt number.
		$payment_exists = OEM_M_Payment::get_payment_by_receipt_number( $receipt_number );
		if ( $payment_exists ) {
			// Regenerate receipt number.
			return self::generate_receipt_number();
		} else {
			return $receipt_number;
		}
	}

	public static function get_access_type( $key, $empty = '-' ) {
		if ( isset( self::access_types()[ $key ] ) ) {
			return self::access_types()[ $key ];
		}

		return $empty;
	}

	public static function access_types() {
		 return array(
			 // 'lifetime'  => esc_html__( 'Lifetime Access', 'online-exam-management' ),
			 // 'limited'   => esc_html__( 'Membership', 'online-exam-management' ),
			 'specified' => esc_html__( 'Exams Based', 'online-exam-management' ),
		 );
	}

	public static function registration_search_fields() {
		return array(
			'reg_number'   => esc_html__( 'Registration No.', 'online-exam-management' ),
			'name'         => esc_html__( 'Name', 'online-exam-management' ),
			'phone'        => esc_html__( 'Phone', 'online-exam-management' ),
			'address'      => esc_html__( 'Address', 'online-exam-management' ),
			'father_name'  => esc_html__( "Father's Name", 'online-exam-management' ),
			'father_phone' => esc_html__( "Father's Phone", 'online-exam-management' ),
			'user_field'   => esc_html__( 'Login Email / Username', 'online-exam-management' ),
		);
	}

	public static function registration_meta_fields() {
		 return array( 'reg_number', 'name', 'phone' );
	}

	public static function lm_valid() {
		return true;
		 $oem_lm    = OEM_LM::get_instance();
		$oem_lm_val = $oem_lm->is_valid();
		if ( isset( $oem_lm_val ) && $oem_lm_val ) {
			return true;
		}
		return false;
	}

	/**
	 * 09 02 2022
	 * get the quesion and answer sheet
	 */
	public static function get_exam_title($exam_id) {
		global $wpdb;
		$query = 'SELECT oe.exam_code, oe.exam_title FROM ' . OEM_EXAMS . ' as oe WHERE oe.ID ='. $exam_id;
		return $wpdb->get_results($wpdb->prepare($query));
	}
	public static function get_question_answer_sheet( $student_id, $exam_id ) {
		global $wpdb;
		$query = 'SELECT q.ID, q.description, q.type, os.user_id, os.exam_id FROM ' . OEM_QUESTIONS .' as q JOIN '. OEM_STUDENTS .' as os ON os.exam_id = q.exam_id WHERE q.exam_id = '. $exam_id;
		return $wpdb->get_results($wpdb->prepare($query));
	}

	public static function get_question_option( $exam_id, $quid ) {
		global $wpdb;
		$query = 'SELECT op.value, op.is_correct FROM ' . OEM_OPTIONS .' as op WHERE op.question_id = '. $quid;
		return $wpdb->get_results($wpdb->prepare($query));
	}
}
