<?php
defined( 'ABSPATH' ) || die();

require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_Helper.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Exam.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Subject.php';

class OEM_Subject {
	public static function fetch_subjects() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		$exam_id = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;

		global $wpdb;

		$page_url = OEM_Helper::get_page_url('exams');

		$query = OEM_M_Subject::fetch_query( $exam_id );

		$query_filter = $query;

		// Grouping.
		$group_by = ' ' . OEM_M_Subject::fetch_query_group_by();

		$query        .= $group_by;
		$query_filter .= $group_by;

		// Searching.
		$condition = '';
		if ( isset( $_POST['search']['value'] ) ) {
			$search_value = sanitize_text_field( $_POST['search']['value'] );
			if ( '' !== $search_value ) {
				$condition .= '' .
				'(sj.subject_name LIKE "%' . $search_value . '%")';

				$query_filter .= ( ' HAVING ' . $condition );
			}
		}

		// Ordering.
		$columns = array( 'sj.subject_name', 'questions_count', 'sj.is_default', 'sj.subject_order' );
		if ( isset( $_POST['order'] ) && isset( $columns[ $_POST['order']['0']['column'] ] ) ) {
			$order_by  = sanitize_text_field( $columns[ $_POST['order']['0']['column'] ] );
			$order_dir = sanitize_text_field( $_POST['order']['0']['dir'] );

			// Sort by subject order.
			if ( 'sj.subject_order' === $order_by ) {
				$order_by = 'sj.subject_order * 1';
			}

			$query_filter .= ' ORDER BY ' . $order_by . ' ' . $order_dir;
		} else {
			$query_filter .= ' ORDER BY sj.subject_order * 1 ASC';
		}

		// Limiting.
		$limit = '';
		if ( -1 != $_POST['length'] ) {
			$start  = absint( $_POST['start'] );
			$length = absint( $_POST['length'] );

			$limit  = ' LIMIT ' . $start . ', ' . $length;
		}

		// Total query.
		$rows_query = OEM_M_Subject::fetch_query_count( $exam_id );

		// Total rows count.
		$total_rows_count = $wpdb->get_var( $rows_query );

		// Filtered rows count.
		if ( $condition ) {
			$filter_rows_count = $wpdb->get_var( $rows_query . ' AND (' . $condition . ')' );
		} else {
			$filter_rows_count = $total_rows_count;
		}

		// Filtered limit rows.
		$filter_rows_limit = $wpdb->get_results( $query_filter . $limit );

		$data = array();
		if ( count( $filter_rows_limit ) ) {
			foreach ( $filter_rows_limit as $row ) {
				// Table columns.
				$data[] = array(
					esc_html( OEM_Helper::stripslashes( $row->subject_name ) ),
					absint( $row->questions_count ),
					OEM_Helper::get_yes_no_text( $row->is_default, true ),
					esc_html( $row->subject_order ),
					'<a class="oem-text-primary" href="' . esc_url( $page_url . "&action=subjects&subject_id=" . $row->ID . '&exam_id=' . $exam_id ) . '"><span class="dashicons dashicons-edit"></span></a>' . ( ! $row->is_default ? '&nbsp;&nbsp;
					<a class="oem-text-danger oem-delete-subject" data-nonce="' . esc_attr( wp_create_nonce( 'delete-subject-' . $row->ID ) ) . '" data-subject="' . esc_attr( $row->ID ) . '" data-exam="' . esc_attr( $row->exam_id ) . '" href="#" data-message-title="' . esc_attr__( 'Please Confirm!', 'online-exam-management' ) . '" data-message-content="' . esc_attr__( 'This will delete the subject. All questions of this subject will be assigned to the default subject.', 'online-exam-management' ) . '" data-cancel="' . esc_attr__( 'Cancel', 'online-exam-management' ) . '" data-submit="' . esc_attr__( 'Confirm', 'online-exam-management' ) . '"><span class="dashicons dashicons-trash"></span></a>' : '' )
				);
			}
		}

		$output = array(
			'draw'            => absint( $_POST['draw'] ),
			'recordsTotal'    => $total_rows_count,
			'recordsFiltered' => $filter_rows_count,
			'data'            => $data,
		);

		echo json_encode( $output );
		die;
	}

	public static function save_subject() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		$subject_id = isset( $_POST['subject_id'] ) ? absint( $_POST['subject_id'] ) : 0;
		$exam_id    = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;

		OEM_Helper::check_nonce( $subject_id ? 'edit-subject-' . $subject_id : 'add-subject' );

		try {
			ob_start();
			global $wpdb;

			$errors = array();

			// Checks if exam exists.
			$exam = OEM_M_Exam::get_exam( $exam_id );
			if ( ! $exam ) {
				throw new Exception( esc_html__( 'Exam not found.', 'online-exam-management' ) );
			}

			if ( $subject_id ) {
				// Checks if subject exists in exam.
				$subject = OEM_M_Subject::fetch_subject( $subject_id, $exam_id );
				if ( ! $subject ) {
					throw new Exception( esc_html__( 'Subject not found.', 'online-exam-management' ) );
				}
			}

			$subject_name  = isset( $_POST['subject_name'] ) ? sanitize_text_field( $_POST['subject_name'] ) : '';
			$subject_order = isset( $_POST['subject_order'] ) ? sanitize_text_field( $_POST['subject_order'] ) : '';
			$is_default    = isset( $_POST['is_default'] ) ? (bool) $_POST['is_default'] : 0;

			if ( empty( $subject_name ) ) {
				$errors['subject_name'] = esc_html__( 'Please specify subject name.', 'online-exam-management' );
			} elseif ( strlen( $subject_name ) > 191 ) {
				$errors['subject_name'] = esc_html__( 'Maximum length cannot exceed 191 characters.', 'online-exam-management' );
			}

			if ( ! is_numeric( $subject_order ) ) {
				$errors['subject_order'] = esc_html__( 'Please provide subject order.', 'online-exam-management' );
			}

			$default_subject = OEM_M_Exam::get_default_subject( $exam_id );
			if ( ! $is_default ) {
				// Check if there is no default subject exists in exam.
				if ( ! $default_subject || ( absint( $default_subject->ID ) === $subject_id ) ) {
					throw new Exception( esc_html__( 'There must be a default subject in the exam.', 'online-exam-management' ) );
				}
			}

			if ( count( $errors ) > 0 ) {
				wp_send_json_error( $errors );
			}

			OEM_Helper::check_buffer();

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}

		try {
			$wpdb->query( 'BEGIN;' );

			// Data to update or insert.
			$data = array(
				'subject_name'  => $subject_name,
				'subject_order' => $subject_order,
				'is_default'    => $is_default,
				'exam_id'       => $exam_id,
			);

			// Checks if update or insert.
			if ( $subject_id ) {
				$data['updated_at'] = OEM_Helper::now();

				$success = $wpdb->update( OEM_SUBJECTS, $data, array( 'ID' => $subject_id ) );

				$message = esc_html__( 'Subject updated successfully.', 'online-exam-management' );
				$reset   = false;

			} else {
				$data['created_at'] = OEM_Helper::now();

				$success    = $wpdb->insert( OEM_SUBJECTS, $data );
				$subject_id = $wpdb->insert_id;

				$message = esc_html__( 'Subject added successfully.', 'online-exam-management' );
				$reset   = true;
			}

			if ( $is_default ) {
				$success = $wpdb->query(
					$wpdb->prepare( 'UPDATE ' . OEM_SUBJECTS . ' SET is_default = 0 WHERE ID != %d AND exam_id = %d', $subject_id, $exam_id )
				);
			}

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			wp_send_json_success( array( 'message' => $message, 'reset' => $reset ) );
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}

	public static function delete_subject() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		try {
			ob_start();
			global $wpdb;

			$exam_id    = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;
			$subject_id = isset( $_POST['subject_id'] ) ? absint( $_POST['subject_id'] ) : 0;

			OEM_Helper::check_nonce( 'delete-subject-' . $subject_id );

			// Checks if subject exists in exam.
			$subject = OEM_M_Subject::fetch_subject( $subject_id, $exam_id );
			if ( ! $subject ) {
				throw new Exception( esc_html__( 'Subject not found.', 'online-exam-management' ) );
			}

			if ( $subject->is_default ) {
				throw new Exception( esc_html__( 'Default subject can not be deleted.', 'online-exam-management' ) );
			}

			OEM_Helper::check_buffer();

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}

		try {
			$wpdb->query( 'BEGIN;' );

			// Assign questions to default subject.
			if ( OEM_M_Subject::get_questions_count( $subject_id ) > 0 ) {
				$default_subject = OEM_M_Exam::get_default_subject( $exam_id );
				if ( ! $default_subject ) {
					throw new Exception( esc_html__( 'Default subject not found.', 'online-exam-management' ) );
				}

				$success = $wpdb->update(
					OEM_QUESTIONS,
					array( 'subject_id' => $default_subject->ID ),
					array( 'subject_id' => $subject_id )
				);
			}

			$success = $wpdb->delete( OEM_SUBJECTS, array( 'ID' => $subject_id ) );

			$message = esc_html__( 'Subject deleted successfully.', 'online-exam-management' );

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			wp_send_json_success( array( 'message' => $message ) );
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}
}
