<?php
defined('ABSPATH') || die();

require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_Helper.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Student.php';

class OEM_Student {
	public static function fetch_exams_students() {
		if (!current_user_can(OEM_ADMIN_CAPABILITY)) {
			die();
		}

		global $wpdb;

		$page_url = OEM_Helper::get_page_url('students');

		$query = OEM_M_Exam::fetch_students_query();

		$query_filter = $query;

		// Grouping.
		$group_by = ' ' . OEM_M_Exam::fetch_query_group_by();

		$query        .= $group_by;
		$query_filter .= $group_by;

		// Searching.
		$condition = '';
		if (isset($_POST['search']['value'])) {
			$search_value = sanitize_text_field($_POST['search']['value']);
			if ('' !== $search_value) {
				$condition .= '' .
					'(ex.exam_title LIKE "%' . $search_value . '%") OR ' .
					'(ex.exam_code LIKE "%' . $search_value . '%") OR ' .
					'(ex.duration LIKE "%' . $search_value . '%")';

				$date_filter = OEM_Helper::filter_by_date($search_value);

				$exam_at        = $date_filter['date'];
				$format_exam_at = $date_filter['format'];

				if ($exam_at && !empty($format_exam_at)) {
					$exam_at = $exam_at->format($format_exam_at);
					$exam_at = ' OR (ex.exam_at LIKE "%' . $exam_at . '%")';

					$condition .= $exam_at;
				}

				$query_filter .= (' HAVING ' . $condition);
			}
		}

		// Ordering.
		$columns = array('ex.exam_title', 'ex.exam_code', 'ex.exam_at', 'ex.duration', 'students_count');
		if (isset($_POST['order']) && isset($columns[$_POST['order']['0']['column']])) {
			$order_by  = sanitize_text_field($columns[$_POST['order']['0']['column']]);
			$order_dir = sanitize_text_field($_POST['order']['0']['dir']);

			$query_filter .= ' ORDER BY ' . $order_by . ' ' . $order_dir;
		} else {
			$query_filter .= ' ORDER BY ex.ID DESC';
		}

		// Limiting.
		$limit = '';
		if (-1 != $_POST['length']) {
			$start  = absint($_POST['start']);
			$length = absint($_POST['length']);

			$limit  = ' LIMIT ' . $start . ', ' . $length;
		}

		// Total query.
		$rows_query = OEM_M_Exam::fetch_students_query_count();

		// Total rows count.
		$total_rows_count = $wpdb->get_var($rows_query);

		// Filtered rows count.
		if ($condition) {
			$filter_rows_count = $wpdb->get_var($rows_query . ' WHERE (' . $condition . ')');
		} else {
			$filter_rows_count = $total_rows_count;
		}

		// Filtered limit rows.
		$filter_rows_limit = $wpdb->get_results($query_filter . $limit);

		$data = array();
		if (count($filter_rows_limit)) {
			foreach ($filter_rows_limit as $row) {
				$students_count = absint($row->students_count);

				if ($students_count > 0) {
					$students_count .= ('&nbsp; - <button type="button" class="btn oem-btn-xs oem-btn-warning oem-students-csv-export-btn" data-exam="' . esc_attr($row->ID) . '">' . esc_html__('Export', 'online-exam-management') . '</button>');
				}

				// Table columns.
				$data[] = array(
					esc_html(OEM_Helper::stripslashes($row->exam_title)),
					esc_html($row->exam_code),
					esc_html(OEM_Helper::get_at_text($row->exam_at)),
					absint($row->duration),
					$students_count,
					'<a class="btn btn-sm oem-btn-primary oem-font-bold" href="' . esc_url($page_url . "&action=list&exam_id=" . $row->ID) . '">' . esc_html__('Manage Students', 'online-exam-management') . '</a>'
				);
			}
		}

		$output = array(
			'draw'            => absint($_POST['draw']),
			'recordsTotal'    => $total_rows_count,
			'recordsFiltered' => $filter_rows_count,
			'data'            => $data,
		);

		echo json_encode($output);
		die;
	}

	public static function fetch_students() {
		if (!current_user_can(OEM_ADMIN_CAPABILITY)) {
			die();
		}

		$exam_id = isset($_POST['exam_id']) ? absint($_POST['exam_id']) : 0;

		global $wpdb;

		$page_url = OEM_Helper::get_page_url('students');

		$query = OEM_M_Student::fetch_query($exam_id);

		$query_filter = $query;

		// Grouping.
		$group_by = ' ' . OEM_M_Student::fetch_query_group_by();

		$query        .= $group_by;
		$query_filter .= $group_by;

		// Searching.
		$condition = '';
		if (isset($_POST['search']['value'])) {
			$search_value = sanitize_text_field($_POST['search']['value']);
			if ('' !== $search_value) {
				$condition .= '' .
					'(s.name LIKE "%' . $search_value . '%") OR ' .
					'(s.roll_number LIKE "%' . $search_value . '%") OR ' .
					'(s.class LIKE "%' . $search_value . '%") OR ' .
					'(s.section LIKE "%' . $search_value . '%") OR ' .
					'(s.phone LIKE "%' . $search_value . '%") OR ' .
					'(s.father_name LIKE "%' . $search_value . '%") OR ' .
					'(s.father_phone LIKE "%' . $search_value . '%") OR ' .
					'(u.user_login LIKE "%' . $search_value . '%") OR ' .
					'(u.user_email LIKE "%' . $search_value . '%")';

				$query_filter .= (' HAVING ' . $condition);
			}
		}

		// Ordering.
		$columns = array('s.name', 's.roll_number', 's.class', 's.section', 's.phone', 's.father_name', 's.father_phone', 'u.user_email', 'u.user_login');
		if (isset($_POST['order']) && isset($columns[$_POST['order']['0']['column']])) {
			$order_by  = sanitize_text_field($columns[$_POST['order']['0']['column']]);
			$order_dir = sanitize_text_field($_POST['order']['0']['dir']);

			$query_filter .= ' ORDER BY ' . $order_by . ' ' . $order_dir;
		} else {
			$query_filter .= ' ORDER BY s.roll_number ASC';
		}

		// Limiting.
		$limit = '';
		if (-1 != $_POST['length']) {
			$start  = absint($_POST['start']);
			$length = absint($_POST['length']);

			$limit  = ' LIMIT ' . $start . ', ' . $length;
		}

		// Total query.
		$rows_query = OEM_M_Student::fetch_query_count($exam_id);

		// Total rows count.
		$total_rows_count = $wpdb->get_var($rows_query);

		// Filtered rows count.
		if ($condition) {
			$filter_rows_count = $wpdb->get_var($rows_query . ' AND (' . $condition . ')');
		} else {
			$filter_rows_count = $total_rows_count;
		}

		// Filtered limit rows.
		$filter_rows_limit = $wpdb->get_results($query_filter . $limit);

		$data = array();
		if (count($filter_rows_limit)) {
			foreach ($filter_rows_limit as $row) {
				// Table columns.
				$data[] = array(
					'<input type="checkbox" class="oem-select-single oem-bulk-students" name="bulk_data[]" value="' . esc_attr($row->ID) . '">',
					esc_html(OEM_Helper::stripslashes($row->name)),
					esc_html(OEM_Helper::get_text($row->roll_number)),
					esc_html(OEM_Helper::stripslashes($row->class, '-')),
					esc_html(OEM_Helper::stripslashes($row->section, '-')),
					esc_html(OEM_Helper::get_text($row->phone, '-')),
					esc_html(OEM_Helper::get_text($row->login_email, '-')),
					esc_html(OEM_Helper::get_text($row->username, '-')),
					esc_html(OEM_Helper::stripslashes($row->father_name, '-')),
					'<img src="'.esc_attr($row->exam_photos).'" width="150" height="150" alt="" >',
					'<a class="oem-text-primary" href="' . esc_url($page_url . "&action=save&student_id=" . $row->ID . '&exam_id=' . $exam_id) . '"><span class="dashicons dashicons-edit"></span></a>&nbsp;&nbsp;
					<a class="oem-text-danger oem-delete-student" data-nonce="' . esc_attr(wp_create_nonce('delete-student-' . $row->ID)) . '" data-student="' . esc_attr($row->ID) . '" data-exam="' . esc_attr($row->exam_id) . '" href="#" data-message-title="' . esc_attr__('Please Confirm!', 'online-exam-management') . '" data-message-content="' . esc_attr__('This will delete the student.', 'online-exam-management') . '" data-cancel="' . esc_attr__('Cancel', 'online-exam-management') . '" data-submit="' . esc_attr__('Confirm', 'online-exam-management') . '"><span class="dashicons dashicons-trash"></span></a>'
				);
			}
		}

		$output = array(
			'draw'            => absint($_POST['draw']),
			'recordsTotal'    => $total_rows_count,
			'recordsFiltered' => $filter_rows_count,
			'data'            => $data,
		);

		echo json_encode($output);
		die;
	}

	public static function save_student() {
		if (!current_user_can(OEM_ADMIN_CAPABILITY)) {
			die();
		}

		$student_id = isset($_POST['student_id']) ? absint($_POST['student_id']) : 0;
		$exam_id    = isset($_POST['exam_id']) ? absint($_POST['exam_id']) : 0;

		OEM_Helper::check_nonce($student_id ? 'edit-student-' . $student_id : 'add-student');

		try {
			ob_start();
			global $wpdb;

			$errors = array();

			// Checks if exam exists.
			$exam = OEM_M_Exam::get_exam($exam_id);
			if (!$exam) {
				throw new Exception(esc_html__('Exam not found.', 'online-exam-management'));
			}

			$user_id = NULL;

			if ($student_id) {
				// Checks if student exists in exam.
				$student = OEM_M_Student::get_student($student_id, $exam_id);
				if (!$student) {
					throw new Exception(esc_html__('Student not found.', 'online-exam-management'));

					$user_id = $student->user_id;
				}
			}

			// Student detail.
			$name         = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
			$roll_number  = isset($_POST['roll_number']) ? sanitize_text_field($_POST['roll_number']) : '';
			$class        = isset($_POST['class']) ? sanitize_text_field($_POST['class']) : '';
			$section      = isset($_POST['section']) ? sanitize_text_field($_POST['section']) : '';
			$phone        = isset($_POST['phone']) ? sanitize_text_field($_POST['phone']) : '';
			$address      = isset($_POST['address']) ? sanitize_text_field($_POST['address']) : '';
			$father_name  = isset($_POST['father_name']) ? sanitize_text_field($_POST['father_name']) : '';
			$father_phone = isset($_POST['father_phone']) ? sanitize_text_field($_POST['father_phone']) : '';
			$gender       = isset($_POST['gender']) ? sanitize_text_field($_POST['gender']) : '';
			$id_number    = isset($_POST['id_number']) ? sanitize_text_field($_POST['id_number']) : '';
			$dob          = !empty( $_POST['dob'] ) ? DateTime::createFromFormat( OEM_Helper::date_format(), sanitize_text_field( $_POST['dob'] ) ) : '';
			$father_phone = isset($_POST['father_phone']) ? sanitize_text_field($_POST['father_phone']) : '';
			$photo        = (isset($_FILES['photo']) && is_array($_FILES['photo'])) ? $_FILES['photo'] : NULL;

			// Login detail.
			$new_or_existing   = isset($_POST['student_new_or_existing']) ? sanitize_text_field($_POST['student_new_or_existing']) : '';
			$existing_username = isset($_POST['existing_username']) ? sanitize_text_field($_POST['existing_username']) : '';
			$disconnect_user   = isset($_POST['disconnect_user']) ? (bool) ($_POST['disconnect_user']) : '';
			$new_login_email   = isset($_POST['new_login_email']) ? sanitize_text_field($_POST['new_login_email']) : '';
			$new_password      = isset($_POST['new_password']) ? sanitize_text_field($_POST['new_password']) : '';
			$username          = isset($_POST['username']) ? sanitize_text_field($_POST['username']) : '';
			$login_email       = isset($_POST['login_email']) ? sanitize_text_field($_POST['login_email']) : '';
			$password          = isset($_POST['password']) ? sanitize_text_field( $_POST['password']) : '';

			// Student detail.
			if ( !empty( $dob ) ) {
				$dob = $dob->format( 'Y-m-d H:i:s' );
			}

			if (empty($name)) {
				$errors['name'] = esc_html__('Please enter student name.', 'online-exam-management');
			} elseif (strlen($name) > 60) {
				$errors['name'] = esc_html__('Maximum length cannot exceed 60 characters.', 'online-exam-management');
			}

			if (!empty($roll_number) && strlen($roll_number) > 40) {
				$errors['roll_number'] = esc_html__('Maximum length cannot exceed 40 characters.', 'online-exam-management');
			}

			if (!empty($class) && strlen($class) > 191) {
				$errors['class'] = esc_html__('Maximum length cannot exceed 191 characters.', 'online-exam-management');
			}

			if (!empty($section) && strlen($section) > 191) {
				$errors['section'] = esc_html__('Maximum length cannot exceed 191 characters.', 'online-exam-management');
			}

			if (!empty($phone) && strlen($phone) > 40) {
				$errors['phone'] = esc_html__('Maximum length cannot exceed 40 characters.', 'online-exam-management');
			}

			if (!empty($father_name) && strlen($father_name) > 60) {
				$errors['father_name'] = esc_html__('Maximum length cannot exceed 60 characters.', 'online-exam-management');
			}

			if (!empty($father_phone) && strlen($father_phone) > 40) {
				$errors['father_phone'] = esc_html__('Maximum length cannot exceed 40 characters.', 'online-exam-management');
			}

			if (isset($photo['tmp_name']) && !empty($photo['tmp_name'])) {
				if (!OEM_Helper::is_valid_file($photo, 'image')) {
					$errors['photo'] = esc_html__('Please provide photo in JPG, JPEG or PNG format.', 'online-exam-management');
				}
			}

			if ($student_id) {
				if (empty($roll_number)) {
					$errors['roll_number'] = esc_html__('Please enter roll number.', 'online-exam-management');
				} else {
					// Check if student already exists with this roll number in exam.
					$student_exists = OEM_M_Student::get_student_by_roll_number($roll_number, $exam_id, $student_id);
					if ($student_exists) {
						$errors['roll_number'] = esc_html__('Roll number already exists in this exam.', 'online-exam-management');
					}
				}
			}

			// Login detail.
			if ('existing_user' === $new_or_existing) {
				if (!$user_id) {
					if (empty($existing_username)) {
						$errors['existing_username'] = esc_html__('Please provide existing username.', 'online-exam-management');
					}
				} else {
					if (empty($new_login_email)) {
						$errors['new_login_email'] = esc_html__('Please provide login email.', 'online-exam-management');
					}
				}
			} elseif ('new_user' === $new_or_existing) {
				if (empty($username)) {
					$errors['username'] = esc_html__('Please provide username.', 'online-exam-management');
				}
				if (empty($login_email)) {
					$errors['login_email'] = esc_html__('Please provide login email.', 'online-exam-management');
				}
				if (!filter_var($login_email, FILTER_VALIDATE_EMAIL)) {
					$errors['login_email'] = esc_html__('Please provide a valid email.', 'online-exam-management');
				}
				if (empty($password)) {
					$errors['password'] = esc_html__('Please provide login password.', 'online-exam-management');
				}
			}

			if (count($errors) > 0) {
				wp_send_json_error($errors);
			}

			OEM_Helper::check_buffer();
		} catch (Exception $exception) {
			wp_send_json_error($exception->getMessage());
		}

		try {
			$wpdb->query('BEGIN;');

			// Student user data.
			$update_student_user_id = NULL;
			if ('existing_user' === $new_or_existing) {
				if (!$user_id) {
					// Existing user.
					$user = get_user_by('login', $existing_username);
					if (!$user) {
						throw new Exception(esc_html__('Username does not exist.', 'online-exam-management'));
					}

					$user_id = $user->ID;

					// Check if user already exists in exam.
					if ($student_id) {
						$user_exists = $wpdb->get_row($wpdb->prepare('SELECT s.ID FROM ' . OEM_STUDENTS . ' as s WHERE s.ID != %d AND s.user_id = %d AND s.exam_id = %d', $student_id, $user_id, $exam_id));
					} else {
						$user_exists = $wpdb->get_row($wpdb->prepare('SELECT s.ID FROM ' . OEM_STUDENTS . ' as s WHERE s.user_id = %d AND s.exam_id = %d', $user_id, $exam_id));
					}

					if ($user_exists) {
						throw new Exception(esc_html__('The user already exists in this exam.', 'online-exam-management'));
					}

					// Check if user is connected to a student record for any other exam.
					if ($student_id) {
						$user_exists = $wpdb->get_row($wpdb->prepare('SELECT s.ID, ex.exam_title, ex.exam_code, ex.exam_at, ex.duration FROM ' . OEM_STUDENTS . ' as s JOIN ' . OEM_EXAMS . ' as ex ON ex.ID = s.exam_id WHERE s.ID != %d AND s.user_id = %d', $student_id, $user_id));
					} else {
						$user_exists = $wpdb->get_row($wpdb->prepare('SELECT s.ID, ex.exam_title, ex.exam_code, ex.exam_at, ex.duration FROM ' . OEM_STUDENTS . ' as s JOIN ' . OEM_EXAMS . ' as ex ON ex.ID = s.exam_id WHERE s.user_id = %d', $user_id));
					}

					if ($user_exists) {
						if ($disconnect_user) {
							$success = $wpdb->update(OEM_STUDENTS, array('user_id' => NULL), array('ID' => $user_exists->ID));
							if (false === $success) {
								throw new Exception($wpdb->last_error);
							}
						} else {
							throw new Exception(
								sprintf(
									wp_kses(
										/* translators: 1: username, 2: exam title, 3: exam code, 4: date and time, 5: number of minutes */
										__('This username <span class="oem-font-bold">%1$s</span> is connected to the exam: <span class="oem-font-bold">%2$s (%3$s) (%4$s - %5$s minutes)</span>.<br>Please disconnect this user from this exam before assigning to a new exam.<br>The username can be associated to only one exam at a time.', 'online-exam-management'),
										array('span' => array('class' => array()), 'br' => array())
									),
									esc_html($user->user_login),
									esc_html(OEM_Helper::stripslashes($user_exists->exam_title)),
									esc_html($user_exists->exam_code),
									esc_html(OEM_Helper::get_at_text($user_exists->exam_at)),
									esc_html($user_exists->duration)
								)
							);
						}
					}
				} else {
					// Update email and password of existing user.
					$user_data = array(
						'ID'         => $user_id,
						'user_email' => $new_login_email,
					);

					if (!empty($new_password)) {
						$user_data['user_pass'] = $new_password;
					}

					$user_id = wp_update_user($user_data);
					if (is_wp_error($user_id)) {
						throw new Exception($user_id->get_error_message());
					}
				}
			} elseif ('new_user' === $new_or_existing) {
				// New user.
				$user_data = array(
					'user_email' => $login_email,
					'user_login' => $username,
					'user_pass'  => $password,
				);

				$user_id = wp_insert_user($user_data);
				if (is_wp_error($user_id)) {
					throw new Exception($user_id->get_error_message());
				}
			} else {
				$user_id = NULL;

				$skip_update_recorded_user_id = true;
			}

			$update_student_user_id = $user_id;

			// Auto generate roll number if empty.
			if (!$student_id && empty($roll_number)) {
				$roll_number = OEM_Helper::generate_roll_number($exam_id);
			}

			// Data to update or insert.
			$data = array(
				'name'         => $name,
				'roll_number'  => $roll_number,
				'class'        => $class,
				'section'      => $section,
				'phone'        => $phone,
				'address'      => $address,
				'father_name'  => $father_name,
				'father_phone' => $father_phone,
				'exam_id'      => $exam_id,
				'user_id'      => $update_student_user_id,
				'dob'          => $dob,
				'gender'       => $gender,
				'id_number'    => $id_number,
			);

			$record_created = false;

			if (!isset($skip_update_recorded_user_id)) {
				$data['record_user_id'] = $update_student_user_id;

				// Create registration if not exists.
				if ($data['record_user_id']) {
					$record_user = OEM_M_Record::find($data['record_user_id']);
					if (!$record_user) {
						$reg_number = OEM_Helper::generate_reg_number();
						OEM_M_Record::save(
							$data['record_user_id'],
							array(
								'reg_number'   => $reg_number,
								'name'         => $name,
								'phone'        => $phone,
								'address'      => $address,
								'father_name'  => $father_name,
								'father_phone' => $father_phone,
							)
						);

						$record_created = true;
					}
				}
			}

			OEM_M_Student::reset_recorded_user_id($update_student_user_id, $exam_id);

			if ($student_id) {
				$data['photo_id'] = $student->photo_id;
			}

			if (!empty($photo)) {
				$photo = media_handle_upload('photo', 0);
				if (is_wp_error($photo)) {
					throw new Exception($photo->get_error_message());
				}
				$data['photo_id'] = $photo;

				if ($record_created) {
					update_user_meta($data['record_user_id'], 'oem_photo_id', $photo);
				}
			}

			// Checks if update or insert.
			if ($student_id) {
				$data['updated_at'] = OEM_Helper::now();

				$success = $wpdb->update(OEM_STUDENTS, $data, array('ID' => $student_id));

				$message = esc_html__('Student updated successfully.', 'online-exam-management');
				$reset   = false;
			} else {
				$data['created_at'] = OEM_Helper::now();

				$success    = $wpdb->insert(OEM_STUDENTS, $data);
				$student_id = $wpdb->insert_id;

				$message = esc_html__('Student added successfully.', 'online-exam-management');
				$reset   = true;
			}

			OEM_Helper::check_buffer();

			if (false === $success) {
				throw new Exception($wpdb->last_error);
			}

			$wpdb->query('COMMIT;');

			wp_send_json_success(array('message' => $message, 'reset' => $reset));
		} catch (Exception $exception) {
			$wpdb->query('ROLLBACK;');
			wp_send_json_error($exception->getMessage());
		}
	}

	public static function delete_student() {
		// OEM_Helper::check_demo_mode();

		if (!current_user_can(OEM_ADMIN_CAPABILITY)) {
			die();
		}

		try {
			ob_start();
			global $wpdb;

			$exam_id    = isset($_POST['exam_id']) ? absint($_POST['exam_id']) : 0;
			$student_id = isset($_POST['student_id']) ? absint($_POST['student_id']) : 0;

			OEM_Helper::check_nonce('delete-student-' . $student_id);

			// Checks if student exists in exam.
			$student = OEM_M_Student::get_student($student_id, $exam_id);

			if (!$student) {
				throw new Exception(esc_html__('Student not found.', 'online-exam-management'));
			}

			OEM_Helper::check_buffer();
		} catch (Exception $exception) {
			wp_send_json_error($exception->getMessage());
		}

		try {
			$wpdb->query('BEGIN;');

			$success = $wpdb->delete(OEM_STUDENTS, array('ID' => $student_id));

			$message = esc_html__('Student deleted successfully.', 'online-exam-management');

			OEM_Helper::check_buffer();

			if (false === $success) {
				throw new Exception($wpdb->last_error);
			}

			$wpdb->query('COMMIT;');

			wp_send_json_success(array('message' => $message));
		} catch (Exception $exception) {
			$wpdb->query('ROLLBACK;');
			wp_send_json_error($exception->getMessage());
		}
	}

	public static function students_csv_export() {
		if (!current_user_can(OEM_ADMIN_CAPABILITY)) {
			die();
		}

		$exam_id = isset($_POST['exam']) ? absint($_POST['exam']) : 0;

		OEM_Helper::check_nonce();

		try {
			ob_start();

			// Get exam students.
			$students = OEM_M_Student::fetch_students($exam_id);

			$fields = array(
				// To be shown in sample CSV file.
				__('Student Name', 'online-exam-management'),
				__('Roll Number', 'online-exam-management'),
				__('Class', 'online-exam-management'),
				__('Section', 'online-exam-management'),
				__('Phone Number', 'online-exam-management'),
				__('Address', 'online-exam-management'),
				__("Father's Name", 'online-exam-management'),
				__("Father's Phone", 'online-exam-management'),
				__('Login Email', 'online-exam-management'),
				__('Username', 'online-exam-management'),
				__('Password', 'online-exam-management'),
			);

			OEM_Helper::check_buffer();

			$filename  = 'students.csv';
			$delimiter = ',';

			$f = fopen('php://memory', 'w');

			fputcsv($f, $fields, $delimiter);

			foreach ($students as $student) {

				$record = array(
					OEM_Helper::stripslashes($student->name),
					OEM_Helper::get_text($student->roll_number),
					OEM_Helper::stripslashes($student->class),
					OEM_Helper::stripslashes($student->section),
					OEM_Helper::get_text($student->phone),
					OEM_Helper::get_text($student->address),
					OEM_Helper::stripslashes($student->father_name),
					OEM_Helper::get_text($student->father_phone),
					OEM_Helper::get_text($student->login_email),
					OEM_Helper::get_text($student->username),
					'',
				);

				fputcsv($f, $record, $delimiter);
			}

			OEM_Helper::export_and_close_csv_file($f, $filename);
		} catch (Exception $exception) {
			wp_send_json_error($exception->getMessage());
		}
	}

	public static function students_sample_csv_export() {
		if (!current_user_can(OEM_ADMIN_CAPABILITY)) {
			die();
		}

		$exam_id = isset($_POST['exam']) ? absint($_POST['exam']) : 0;

		OEM_Helper::check_nonce();

		try {
			ob_start();

			$fields = array(
				// To be shown in sample CSV file.
				__('Student Name', 'online-exam-management'),
				__('Roll Number', 'online-exam-management'),
				__('Class', 'online-exam-management'),
				__('Section', 'online-exam-management'),
				__('Phone Number', 'online-exam-management'),
				__('Address', 'online-exam-management'),
				__("Father's Name", 'online-exam-management'),
				__("Father's Phone", 'online-exam-management'),
				__('Login Email', 'online-exam-management'),
				__('Username', 'online-exam-management'),
				__('Password', 'online-exam-management'),
			);

			OEM_Helper::check_buffer();

			$filename  = 'students_sample.csv';
			$delimiter = ',';

			$f = fopen('php://memory', 'w');

			fputcsv($f, $fields, $delimiter);

			$sample_record = array(
				// To be shown in sample CSV file.
				'Andrew Smith', // Sample student name.
				'1',
				'10th',
				'A',
				'9999999991',
				'101, Sample Address',
				'John Smith', // Sample father's name.
				'9999999992',
				'andrew_smith@gmail.com',
				'andrew',
				'123456',
			);

			fputcsv($f, $sample_record, $delimiter);

			OEM_Helper::export_and_close_csv_file($f, $filename);
		} catch (Exception $exception) {
			wp_send_json_error($exception->getMessage());
		}
	}

	public static function bulk_import_students() {
		if (!current_user_can(OEM_ADMIN_CAPABILITY)) {
			die();
		}

		$to_exam_id = isset($_POST['exam_id']) ? absint($_POST['exam_id']) : 0;

		OEM_Helper::check_nonce('bulk-import-students-' . $to_exam_id);

		try {
			ob_start();
			global $wpdb;

			$errors = array();

			// Checks if exam exists.
			$to_exam = OEM_M_Exam::get_exam_title($to_exam_id);
			if (!$to_exam) {
				throw new Exception(esc_html__('Exam not found.', 'online-exam-management'));
			}

			$import_from = isset($_POST['import_from']) ? sanitize_text_field($_POST['import_from']) : '';

			if (!in_array($import_from, array('exam', 'csv'))) {
				throw new Exception(esc_html__('Please select from where to import students.', 'online-exam-management'));
			}

			// Get "to exam" students.
			$to_exam_students = OEM_M_Student::fetch_students($to_exam_id);

			// Get roll_numbers, user_ids and usernames of students.
			$to_exam_students_data = array(
				'roll_numbers' => array(),
				'user_ids'     => array(),
				'usernames'    => array(),
				'login_emails' => array()
			);

			foreach ($to_exam_students as $student) {
				if ($student->roll_number && !in_array($student->roll_number, $to_exam_students_data['roll_numbers'])) {
					array_push($to_exam_students_data['roll_numbers'], $student->roll_number);
				}
				if ($student->user_id && !in_array($student->user_id, $to_exam_students_data['user_ids'])) {
					array_push($to_exam_students_data['user_ids'], $student->user_id);
				}
				if ($student->username && !in_array($student->username, $to_exam_students_data['usernames'])) {
					array_push($to_exam_students_data['usernames'], $student->username);
				}
				if ($student->login_email && !in_array($student->login_email, $to_exam_students_data['login_emails'])) {
					array_push($to_exam_students_data['login_emails'], $student->login_email);
				}
			}

			if ('exam' === $import_from) {
				// Import from exam.
				$from_exam_id    = isset($_POST['exam']) ? absint($_POST['exam']) : 0;
				$disconnect_user = isset($_POST['disconnect_user_exam']) ? (bool) ($_POST['disconnect_user_exam']) : '';

				if (!$from_exam_id) {
					$errors['keyword'] = esc_html__('Please select an exam from where to import the students.', 'online-exam-management');
					wp_send_json_error($errors);
				} else {
					$from_exam = OEM_M_Exam::get_exam_title($from_exam_id);
					if (!$from_exam || ($from_exam->ID === $to_exam->ID)) {
						throw new Exception(esc_html__('Exam not found.', 'online-exam-management'));
					}
				}

				// Get "from exam" students.
				$from_exam_students = OEM_M_Student::fetch_students($from_exam_id);

				foreach ($from_exam_students as $student) {
					// For each "from exam" student, check if roll number already exists in "to exam".
					if (in_array($student->roll_number, $to_exam_students_data['roll_numbers'])) {
						throw new Exception(
							sprintf(
								/* translators: 1: roll number, 2: exam title, 3: exam code */
								esc_html__('Roll number %1$s already exist in exam: %2$s (%3$s).', 'online-exam-management'),
								esc_html($student->roll_number),
								esc_html(OEM_Helper::stripslashes($to_exam->exam_title)),
								esc_html($to_exam->exam_code)
							)
						);
					}

					// For each "from exam" student, check if username already exists in "to exam".
					if (in_array($student->user_id, $to_exam_students_data['user_ids'])) {
						throw new Exception(
							sprintf(
								/* translators: 1: username, 2: exam title, 3: exam code */
								esc_html__('Username %1$s already exist in exam: %2$s (%3$s).', 'online-exam-management'),
								esc_html($student->username),
								esc_html(OEM_Helper::stripslashes($to_exam->exam_title)),
								esc_html($to_exam->exam_code)
							)
						);
					}
				}
			} else {
				// Import from CSV.
				$csv             = (isset($_FILES['csv']) && is_array($_FILES['csv'])) ? $_FILES['csv'] : NULL;
				$disconnect_user = isset($_POST['disconnect_user_csv']) ? (bool) ($_POST['disconnect_user_csv']) : '';

				if (isset($csv['tmp_name']) && !empty($csv['tmp_name'])) {
					if (!OEM_Helper::is_valid_file($csv, 'csv')) {
						$errors['csv'] = esc_html__('Invalid csv file.', 'online-exam-management');
						wp_send_json_error($errors);
					}
				} else {
					$errors['csv'] = esc_html__('Please provide a csv file.', 'online-exam-management');
					wp_send_json_error($errors);
				}

				// Get "csv students.
				$csv_students = array();

				// Collect roll numbers and usernames to check for duplication.
				$csv_roll_numbers = array();
				$csv_usernames    = array();

				$csv_file = fopen($csv['tmp_name'], 'r');

				fgetcsv($csv_file);

				$row = 1;
				while ($line = fgetcsv($csv_file)) {
					$row++;

					$name         = sanitize_text_field($line[0]);
					$roll_number  = sanitize_text_field($line[1]);
					$class        = sanitize_text_field($line[2]);
					$section      = sanitize_text_field($line[3]);
					$phone        = sanitize_text_field($line[4]);
					$address      = sanitize_text_field($line[5]);
					$father_name  = sanitize_text_field($line[6]);
					$father_phone = sanitize_text_field($line[7]);
					$login_email  = sanitize_text_field($line[8]);
					$username     = sanitize_text_field($line[9]);
					$password     = $line[10];

					if (!empty($roll_number)) {
						array_push($csv_roll_numbers, $roll_number);
					}

					if (!empty($username)) {
						array_push($csv_usernames, $username);
					}

					array_push(
						$csv_students,
						array(
							'row'          => $row,
							'name'         => $name,
							'roll_number'  => $roll_number,
							'class'        => $class,
							'section'      => $section,
							'phone'        => $phone,
							'address'      => $address,
							'father_name'  => $father_name,
							'father_phone' => $father_phone,
							'login_email'  => $login_email,
							'username'     => $username,
							'password'     => $password,
						)
					);


					// Check duplicate roll numbers and usernames.
					if (OEM_Helper::array_has_duplicates($csv_roll_numbers)) {
						throw new Exception(esc_html__('All roll numbers should be unqiue.', 'online-exam-management'));
					}
					if (OEM_Helper::array_has_duplicates($csv_usernames)) {
						throw new Exception(esc_html__('All usernames should be unqiue.', 'online-exam-management'));
					}

					foreach ($csv_students as $key => $student) {
						$name         = $student['name'];
						$roll_number  = $student['roll_number'];
						$class        = $student['class'];
						$section      = $student['section'];
						$phone        = $student['phone'];
						$address      = $student['address'];
						$father_name  = $student['father_name'];
						$father_phone = $student['father_phone'];
						$login_email  = $student['login_email'];
						$username     = $student['username'];
						$password     = $student['password'];

						// Student detail.
						if (empty($name)) {
							throw new Exception(esc_html__('Please enter student name.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 1));
						} elseif (strlen($name) > 60) {
							throw new Exception(esc_html__('Maximum length cannot exceed 60 characters.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 1));
						}

						if (empty($roll_number)) {
							throw new Exception(esc_html__('Please enter roll number.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 2));
						} elseif (strlen($roll_number) > 40) {
							throw new Exception(esc_html__('Maximum length cannot exceed 40 characters.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 2));
						}

						if (!empty($class) && strlen($class) > 191) {
							throw new Exception(esc_html__('Maximum length cannot exceed 191 characters.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 3));
						}

						if (!empty($section) && strlen($section) > 191) {
							throw new Exception(esc_html__('Maximum length cannot exceed 191 characters.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 4));
						}

						if (!empty($phone) && strlen($phone) > 40) {
							throw new Exception(esc_html__('Maximum length cannot exceed 40 characters.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 5));
						}

						if (!empty($father_name) && strlen($father_name) > 60) {
							throw new Exception(esc_html__('Maximum length cannot exceed 60 characters.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 6));
						}

						if (!empty($father_phone) && strlen($father_phone) > 40) {
							throw new Exception(esc_html__('Maximum length cannot exceed 40 characters.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 7));
						}

						// For each "csv" student, check if roll number already exists in "to exam".
						if (in_array($student['roll_number'], $to_exam_students_data['roll_numbers'])) {
							throw new Exception(
								sprintf(
									/* translators: 1: roll number, 2: exam title, 3: exam code */
									esc_html__('Roll number %1$s already exist in exam: %2$s (%3$s).', 'online-exam-management'),
									esc_html($student['roll_number']),
									esc_html(OEM_Helper::stripslashes($to_exam->exam_title)),
									esc_html($to_exam->exam_code)
								)
							);
						}

						// For each "csv" student, check if username already exists in "to exam".
						if (in_array($student['username'], $to_exam_students_data['usernames'])) {
							throw new Exception(
								sprintf(
									/* translators: 1: username, 2: exam title, 3: exam code */
									esc_html__('Username %1$s already exist in exam: %2$s (%3$s).', 'online-exam-management'),
									esc_html($student['username']),
									esc_html(OEM_Helper::stripslashes($to_exam->exam_title)),
									esc_html($to_exam->exam_code)
								)
							);
						}
						// var_dump($login_email, $username, $password);
						// Login detail.
						$csv_students[$key]['user_id'] = '';

						// Disconnect student's user_id.
						$csv_students[$key]['disconnect_student_user_id'] = '';

						if (!empty($username)) {
							// Check if existing user.
							$user = get_user_by('login', $username);
							if ($user) {
								// If existing user.
								$user_id = $user->ID;

								// Check if user is connected to a student record for any other exam.
								$user_exists = $wpdb->get_row($wpdb->prepare('SELECT s.ID, ex.exam_title, ex.exam_code, ex.exam_at, ex.duration FROM ' . OEM_STUDENTS . ' as s JOIN ' . OEM_EXAMS . ' as ex ON ex.ID = s.exam_id WHERE s.user_id = %d', $user_id));

								if ($user_exists) {
									if ($disconnect_user) {
										$csv_students[$key]['disconnect_student_user_id'] = $user_exists->ID;
									} else {
										throw new Exception(
											sprintf(
												wp_kses(
													/* translators: 1: username, 2: exam title, 3: exam code, 4: date and time, 5: number of minutes */
													__('This username <span class="oem-font-bold">%1$s</span> is connected to the exam: <span class="oem-font-bold">%2$s (%3$s) (%4$s - %5$s minutes)</span>.<br>Please disconnect this user from this exam before assigning to a new exam.<br>The username can be associated to only one exam at a time.', 'online-exam-management'),
													array('span' => array('class' => array()), 'br' => array())
												),
												esc_html($username),
												esc_html(OEM_Helper::stripslashes($user_exists->exam_title)),
												esc_html($user_exists->exam_code),
												esc_html(OEM_Helper::get_at_text($user_exists->exam_at)),
												esc_html($user_exists->duration)
											)
										);
									}
								}

								// We will check if username already exists using user_id later and also update login email and password if provided.

								$csv_students[$key]['user_id'] = $user_id;
							} else {
								// If not existing user, check if login email and password is provided.
								if (empty($login_email)) {
									throw new Exception(esc_html__('Please provide login email.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 8));
								} elseif (!filter_var($login_email, FILTER_VALIDATE_EMAIL)) {
									throw new Exception(esc_html__('Please provide a valid email.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 8));
								}

								if (empty($password)) {
									throw new Exception(esc_html__('Please provide login password.', 'online-exam-management') . OEM_Helper::get_csv_error_msg($row, 10));
								}
							}
						}
					}
				}
			}

			fclose($csv_file);

			OEM_Helper::check_buffer();
		} catch (Exception $exception) {
			wp_send_json_error($exception->getMessage());
		}

		if (count($errors) > 0) {
			wp_send_json_error($errors);
		}

		try {
			$wpdb->query('BEGIN;');

			if ('exam' === $import_from) {
				// Import from exam.

				// Disconnect student login accounts from exam.
				$wpdb->update(OEM_STUDENTS, array('user_id' => NULL), array('exam_id' => $from_exam_id));

				// For each "from exam" student, insert student record in "to exam" along with user_id.
				foreach ($from_exam_students as $student) {
					// Check if user is connected to a student record for any other exam.
					$user_exists = $wpdb->get_row($wpdb->prepare('SELECT s.ID, ex.exam_title, ex.exam_code, ex.exam_at, ex.duration FROM ' . OEM_STUDENTS . ' as s JOIN ' . OEM_EXAMS . ' as ex ON ex.ID = s.exam_id WHERE s.user_id = %d AND ex.ID != %d', $student->user_id, $from_exam_id));

					if ($user_exists) {
						if ($disconnect_user) {
							$success = $wpdb->update(OEM_STUDENTS, array('user_id' => NULL), array('ID' => $user_exists->ID));
							if (false === $success) {
								throw new Exception($wpdb->last_error);
							}
						} else {
							throw new Exception(
								sprintf(
									wp_kses(
										/* translators: 1: username, 2: exam title, 3: exam code, 4: date and time, 5: number of minutes */
										__('This username <span class="oem-font-bold">%1$s</span> is connected to the exam: <span class="oem-font-bold">%2$s (%3$s) (%4$s - %5$s minutes)</span>.<br>Please disconnect this user from this exam before assigning to a new exam.<br>The username can be associated to only one exam at a time.', 'online-exam-management'),
										array('span' => array('class' => array()), 'br' => array())
									),
									esc_html($student->username),
									esc_html(OEM_Helper::stripslashes($user_exists->exam_title)),
									esc_html($user_exists->exam_code),
									esc_html(OEM_Helper::get_at_text($user_exists->exam_at)),
									esc_html($user_exists->duration)
								)
							);
						}
					}

					// Data to insert.
					$data = array(
						'name'           => $student->name,
						'roll_number'    => $student->roll_number,
						'class'          => $student->class,
						'section'        => $student->section,
						'phone'          => $student->phone,
						'address'        => $student->address,
						'father_name'    => $student->father_name,
						'father_phone'   => $student->father_phone,
						'exam_id'        => $to_exam_id,
						'user_id'        => $student->user_id,
						'record_user_id' => $student->user_id,
						'photo_id'       => $student->photo_id,
						'created_at'     => OEM_Helper::now(),
					);

					// Create registration if not exists.
					if ($student->user_id) {
						$record_user = OEM_M_Record::find($student->user_id);
						if (!$record_user) {
							$reg_number = OEM_Helper::generate_reg_number();
							OEM_M_Record::save(
								$student->user_id,
								array(
									'reg_number'   => $reg_number,
									'name'         => $student->name,
									'phone'        => $student->phone,
									'address'      => $student->address,
									'father_name'  => $student->father_name,
									'father_phone' => $student->father_phone,
								)
							);
						}
					}

					OEM_M_Student::reset_recorded_user_id($student->user_id, $to_exam_id);

					$success = $wpdb->insert(OEM_STUDENTS, $data);

					if (false === $success) {
						throw new Exception($wpdb->last_error);
					}

					$student_id = $wpdb->insert_id;
				}
			} else {
				// Import from CSV.

				// For each "csv" student, insert student record in "to exam" along with user_id.
				foreach ($csv_students as $student) {
					// If user is connected to a student record for any other exam, then disconnect the user.
					if ($student['disconnect_student_user_id']) {
						$success = $wpdb->update(OEM_STUDENTS, array('user_id' => NULL), array('ID' => $student['disconnect_student_user_id']));
						if (false === $success) {
							throw new Exception($wpdb->last_error);
						}
					}

					// Data to insert.
					$data = array(
						'name'           => $student['name'],
						'roll_number'    => $student['roll_number'],
						'class'          => $student['class'],
						'section'        => $student['section'],
						'phone'          => $student['phone'],
						'address'        => $student['address'],
						'father_name'    => $student['father_name'],
						'father_phone'   => $student['father_phone'],
						'exam_id'        => $to_exam_id,
						'created_at'     => OEM_Helper::now(),
					);

					// Insert or update user depending on if user_id is set or not set.
					$username    = $student['username'];
					$login_email = $student['login_email'];
					$password    = $student['password'];

					// Check if existing user or new user.
					if ($student['user_id']) {
						$user_id = $student['user_id'];

						$user_data = array('ID' => $user_id);

						// Update email and password of existing user.
						if (!empty($login_email)) {
							$user_data['user_email'] = $login_email;
						}
						if (!empty($password)) {
							$user_data['user_pass'] = $password;
						}

						// Update email and password of existing user if either email or password or both are provided.
						if (count($user_data) > 1) {
							$user_id = wp_update_user($user_data);
							if (is_wp_error($user_id)) {
								throw new Exception($user_id->get_error_message());
							}
						}

						$data['user_id']        = $user_id;
						$data['record_user_id'] = $user_id;

						// Create registration if not exists.
						if ($data['record_user_id']) {
							$record_user = OEM_M_Record::find($data['record_user_id']);
							if (!$record_user) {
								$reg_number = OEM_Helper::generate_reg_number();
								OEM_M_Record::save(
									$data['record_user_id'],
									array(
										'reg_number'   => $reg_number,
										'name'         => $student['name'],
										'phone'        => $student['phone'],
										'address'      => $student['address'],
										'father_name'  => $student['father_name'],
										'father_phone' => $student['father_phone'],
									)
								);
							}
						}

						OEM_M_Student::reset_recorded_user_id($user_id, $to_exam_id);
					} else {
						// New user, insert if username, login email and password is provided.
						if (!empty($username)) {
							$user_data = array(
								'user_email' => $login_email,
								'user_login' => $username,
								'user_pass'  => $password,
							);

							$user_id = wp_insert_user($user_data);
							if (is_wp_error($user_id)) {
								throw new Exception($user_id->get_error_message());
							}

							// Assign user_id.
							$data['user_id']        = $user_id;
							$data['record_user_id'] = $user_id;

							// Create registration if not exists.
							if ($data['record_user_id']) {
								$record_user = OEM_M_Record::find($data['record_user_id']);
								if (!$record_user) {
									$reg_number = OEM_Helper::generate_reg_number();
									OEM_M_Record::save(
										$data['record_user_id'],
										array(
											'reg_number'   => $reg_number,
											'name'         => $student['name'],
											'phone'        => $student['phone'],
											'address'      => $student['address'],
											'father_name'  => $student['father_name'],
											'father_phone' => $student['father_phone'],
										)
									);
								}
							}

							OEM_M_Student::reset_recorded_user_id($user_id, $to_exam_id);
						}
					}

					$success = $wpdb->insert(OEM_STUDENTS, $data);

					if (false === $success) {
						throw new Exception($wpdb->last_error);
					}

					$student_id = $wpdb->insert_id;
				}
			}

			OEM_Helper::check_buffer();

			if (false === $success) {
				throw new Exception($wpdb->last_error);
			}

			$message = esc_html__('Students imported successfully.', 'online-exam-management');

			$wpdb->query('COMMIT;');

			wp_send_json_success(array('message' => $message));
		} catch (Exception $exception) {
			$wpdb->query('ROLLBACK;');
			wp_send_json_error($exception->getMessage());
		}
	}
}
